package com.jhscale.meter.protocol.print.image;

import com.jhscale.common.model.inter.JSONModel;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.print.dither.ImageBuffer;

import java.io.File;
import java.io.IOException;
import java.util.Objects;

/**
 * @author lie_w
 * @title: ImageBulider
 * @projectName jhscale-agreement
 * @description: 图片构造器
 * @date 2021/3/2923:20
 */
public abstract class ImageBuilder<T extends ImageBuilder> implements JSONModel {

    // 图片处理器
    private ImageProcess imageProcess;

    // 源参数信息
    private File source;

    // 源文件资源信息
    private ImageBuffer sourceBuffer;

    // 保存点图文件
    private File target;

    // 目标资源信息
    private ImageBuffer targetBuffer;

    /**
     * @description: 添加源文件
     **/
    public T addSource(File source) {
        this.source = source;
        return (T) this;
    }

    /**
     * @description: 添加源资源信息
     **/
    public T addSource(ImageBuffer sourceBuffer) {
        this.sourceBuffer = sourceBuffer;
        return (T) this;
    }

    /**
     * @description: 添加图片执行器
     **/
    public T addImageProcess(ImageProcess imageProcess) {
        this.imageProcess = imageProcess;
        return (T) this;
    }

    /**
     * @description: 添加目标保存文件
     **/
    public T addTarget(File target) {
        this.target = target;
        return (T) this;
    }

    /**
     * @description: 检查参数内容
     **/
    protected T check() throws MeterException {
        if (Objects.isNull(this.source) && Objects.isNull(this.sourceBuffer))
            throw new MeterException(MeterStateEnum.图片源文件为空);
        if (Objects.isNull(this.imageProcess))
            throw new MeterException(MeterStateEnum.图片执行器不存在);
        return (T) this;
    }

    /**
     * @description: 预处理
     **/
    protected T before() throws MeterException {
        // 参数检查
        this.check();
        if (Objects.isNull(this.sourceBuffer)) {
            // 读取图片
            this.sourceBuffer = imageProcess.readImage(this.source);
        }
        return (T) this;
    }

    /**
     * @description: 构建图片处理内容
     **/
    public T build() throws MeterException {
        this.before();
        this.targetBuffer = this.construct();
        this.after();
        return (T) this;
    }

    /**
     * @description: 实际构建方式
     **/
    protected abstract ImageBuffer construct();

    /**
     * @description: 结束前处理
     **/
    protected T after() throws MeterException {
        // 检查保存文件以及保存目录
        if (Objects.nonNull(this.target) && Objects.nonNull(this.targetBuffer)) {
            File parentFile = this.target.getParentFile();
            if (!parentFile.exists()) parentFile.mkdirs();
            try {
                if (!target.exists()) target.createNewFile();
            } catch (IOException e) {
                throw new MeterException(MeterStateEnum.创建保存文件失败);
            }
            imageProcess.writeImage(targetBuffer, target);
        }
        return (T) this;
    }

    protected ImageProcess getImageProcess() {
        return imageProcess;
    }

    /**
     * @description: 源文件
     **/
    public File getSource() {
        return source;
    }

    /**
     * @description: 源资源
     **/
    public ImageBuffer getSourceBuffer() {
        return sourceBuffer;
    }

    /**
     * @description: 目标文件
     **/
    public File getTarget() {
        return target;
    }

    /**
     * @description: 目标资源
     **/
    public ImageBuffer getTargetBuffer() {
        return targetBuffer;
    }
}
