package com.jhscale.meter.seal;

import com.jhscale.common.utils.SystemtUtils;
import com.jhscale.meter.auncel.AuncelUtils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.PortManager;
import com.jhscale.meter.io.control.DeviceControl;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.model.device.Device;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.seal.cmd.*;
import com.jhscale.meter.seal.entity.SealRequest;
import com.jhscale.meter.seal.entity.SealResponse;
import com.jhscale.meter.seal.entity.inner.Offset;
import com.jhscale.meter.seal.entity.inner.SealSign;
import com.jhscale.meter.utils.ByteUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;

import static com.jhscale.meter.utils.Constant.TOP;


/**
 * @author wang lie
 * @version 1.0
 * @projectName meter-jar
 * @title: SealCal
 * @description: TODO
 * @date 2024/9/5 14:51
 */
public class SealManager implements DeviceClientEventListener {

    // 偏移量
    private Offset offset;

    // 通讯器
    private PortManager portManager;

    // 循环启动器
    private Timer sealTimer;

    // 请求超时间
    private long timeout = 500;

    // 重试次数
    private int recount = 3;

    // 等待结果休眠时间
    private long cycle_timeout = 10;

    // 等待周期
    private int cycle_count = 100;

    private String cache = "";

    // 请求对象
    private SealRequest link_request;

    // 响应对象
    private SealResponse link_response;

    // 蓝牙关闭通知
    private SealBLENotify bleNotify;

    private boolean ble = false;

    private SealManager() {
    }

    private static class SingleSealManager {
        private static final SealManager SINGLETON = new SealManager();
    }

    public static SealManager getInstance() {
        return SealManager.SingleSealManager.SINGLETON;
    }

    /**
     * @description: 有效性检查
     * @date: 2024-09-09 09:33:04
     **/
    public boolean validate() {
        return Objects.nonNull(this.portManager);
    }

    public boolean Init_Manager(DeviceControl control, Device device) throws MeterException {
        if (this.portManager != null) {
            this.close();
        }

        if (Objects.isNull(control))
            throw new MeterException(MeterStateEnum.控制器不存在);
        if (Objects.isNull(device))
            throw new MeterException(MeterStateEnum.通讯设备不存在);

        this.Execute_Reset();
        this.offset = null;
        this.portManager = new PortManager(control, device, this);
        boolean result = this.portManager.openPort();

        if (result) {
            // 每隔30分钟握手一次
            this.sealTimer = new Timer();
            this.sealTimer.scheduleAtFixedRate(new TimerTask() {
                @Override
                public void run() {
                    handShake();
                }
            }, 0, 3600000);
            SystemtUtils.sleep(2);
        }
        return result;
    }

    /**
     * @description: 关闭方法
     * @date: 2024-08-16 17:15:28
     **/
    public SealManager close() throws MeterException {
        if (this.portManager != null) {
            this.portManager.closePort();
            this.portManager = null;
        }
        if (this.sealTimer != null) {
            this.sealTimer.cancel();
            this.sealTimer = null;
        }
        this.Execute_Reset();
        return this;
    }

    /**
     * @description: 握手
     * @date: 2024-09-05 15:02:28
     **/
    public synchronized boolean handShake() {
        if (this.link_request != null || this.ble) return false;
        this.offset = null;
        SealResponse sealResponse = null;
        HandShakeResponse response = null;
        try {
            sealResponse = this.execute(new HandShakeRequest());
            response = (HandShakeResponse) sealResponse;
        } catch (Exception e) {
            System.err.printf("%s error%n", Objects.nonNull(sealResponse) ? sealResponse.getClass().getName() : null);
        }
        if (response == null) return false;
        this.offset = new Offset(response.offset1(), response.offset2());
        if (GlobalPara.getInstance().isRunLog())
            System.out.printf("握手响应: %s%n偏移量： %s%n", response.toJSON(), this.offset.toJSON());
        return true;
    }

    private ObtainStatusResponse statusResponse = null;

    /**
     * @description: 读取铅封状态
     * @date: 2024-09-05 17:41:52
     */
    public ObtainStatusResponse status() {
        if (this.portManager == null || this.offset == null) return null;
        if (!this.ble) {
            try {
                ObtainStatusResponse response = this.execute(new ObtainStatusRequest(this.offset));
                this.statusResponse = Objects.nonNull(response) ? response : this.statusResponse;
            } catch (Exception e) {
            }
        }
        return this.statusResponse;
    }

    /**
     * @description: 打开蓝牙
     * @date: 2024-09-05 20:58:04
     **/
    public OpenBLEResponse openBLE(String unique, SealBLENotify notify) {
        if (this.portManager == null || this.offset == null) return null;
        this.bleNotify = notify;
        try {
            return this.execute(new OpenBLERequest(this.offset).setUnique(unique));
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * @description: 请求对象重置
     **/
    public SealManager Execute_Reset() {
        this.link_request = null;
        this.link_response = null;
        this.cache = "";
        this.ble = false;
        return this;
    }

    public Offset getOffset() {
        return offset;
    }

    /**
     * @description: 获取发送指令
     * @date: 2024-09-05 19:33:51
     **/
    public SealRequest assembly_command(SealRequest request) throws MeterException {
        return request.execute();
    }

    /**
     * @description: 解析发送内容
     * @date: 2024-09-05 19:35:15
     **/
    public SealResponse disassembly_command(Offset offset, String command) throws MeterException {
        return (Objects.nonNull(this.link_request) ? new SealResponse(offset, this.link_request, command) : new SealResponse(offset, command)).execute();
    }

    /**
     * @description: 发送请求 数据
     **/
    public synchronized <T extends SealRequest, U extends SealResponse> U execute(SealRequest<T, U> request) {
        if (this.ble) return null;
        long timestamp = System.currentTimeMillis();

        SealResponse response = null;
        try {
            String command = this.assembly_command(request).over();

            if (GlobalPara.getInstance().isRunLog())
                System.out.printf("%s 发送数据包： %s %s%n", timestamp, request.getClass().getName(), request.toJSON());

            this.link_request = request;
            this.link_response = null;

            int recount = 0;
            while (recount < this.recount) {

                if (GlobalPara.getInstance().isRunLog())
                    System.out.printf("%s 串口发送 [%s] 内容： 原始内容：[ %s] 加密内容：[ %s] 完整数据包：[ %s]%n", timestamp, (Objects.nonNull(this.offset) ? this.offset.toJSON() : "OFFSET_Null"),
                            AuncelUtils.hexPrint(request.getSource_inner().toString()), AuncelUtils.hexPrint(request.getCryption_inner()), AuncelUtils.hexPrint(command));

                // 发送数据
                this.portManager.writeDataImmediately(ByteUtils.fromHexString(command));

                int cycle_count = 0;
                while (cycle_count < this.cycle_count) {
                    this.cycle_sleep();
                    if (this.link_response != null) {
                        response = this.link_response;
                        break;
                    }
                    cycle_count++;
                }

                if (response != null) break;
                recount++;
            }

        } catch (MeterException e) {
            System.err.printf("SealManager_Err: %s%n", e.getMessage());
        }

        return (U) response;
    }

    /**
     * @description: 等待周期循环
     **/
    private void cycle_sleep() {
        try {
            Thread.sleep(this.cycle_timeout);
        } catch (InterruptedException e) {
        }
    }

    /**
     * @description: 连接监听
     */
    @Override
    public void onClientEvent(byte[] bytes) throws MeterException {
        long timestamp = System.currentTimeMillis();

        String accept = cache + ByteUtils.bytes2HexString(bytes);

        while (StringUtils.isNotBlank(accept)) {
            int start_index = ByteUtils.indexOf(accept, TOP);
            if (start_index == -1) {
                cache = "";
                return;
            }
            accept = accept.substring(start_index);

            // 依据标志位获取长度
            SealSign sealSign = new SealSign(accept.substring(2, 4));
            int length = sealSign.length();

            if (accept.length() < length) {
                cache = accept;
                return;
            } else {
                String command = accept.substring(0, length);
                accept = accept.substring(length);

                if (GlobalPara.getInstance().isRunLog())
                    System.out.printf("%s 串口监听 有效数据包 内容 %s%n", timestamp, AuncelUtils.hexPrint(command));

                SealResponse sealResponse = this.disassembly_command(this.offset, command);

                if (sealResponse == null) {
                    System.err.printf("数据包解析失败：[%s]%n", command);
                    continue;
                } else {
                    if (GlobalPara.getInstance().isRunLog())
                        System.out.printf("%s 接受数据包： %s %s%n", timestamp, sealResponse.getClass().getName(), sealResponse.toJSON());
                }

                if (sealResponse instanceof OpenBLEResponse && this.bleNotify != null) {
                    this.ble = ((OpenBLEResponse) sealResponse).getResult() == 1;
                    this.bleNotify.notify((OpenBLEResponse) sealResponse);
                }
                link_response = sealResponse;
            }
        }
    }

    /**
     * @description: 连接异常监听
     */
    @Override
    public void onClientEventExp(MeterException e) {
        e.printStackTrace();
    }
}
