package com.jhscale.meter.seal.listener;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.listener.DeviceClientEventListener;
import com.jhscale.meter.log.JLog;
import com.jhscale.meter.protocol.*;
import com.jhscale.meter.protocol.ad.em.CommunicationState;
import com.jhscale.meter.protocol.ad.entity.ADPackDisassemblyRequest;
import com.jhscale.meter.protocol.ad.entity.ADPackDisassemblyResponse;
import com.jhscale.meter.protocol.ad.send.ADAsynEvent;
import com.jhscale.meter.protocol.ad.send.ADReadAsynThread;
import com.jhscale.meter.protocol.entity.ProtocolEntity;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.CodeUtils;

import java.util.*;

/**
 * @author lie_w
 * @title: DefaultDeviceClientEventListener
 * @projectName meter-jar
 * @description: AD连接 监听器
 * @date 2021/6/1112:49
 */
public class SealClientEventListener implements DeviceClientEventListener, ADAsynEvent {

    // 通讯状态
    CommunicationState communication = CommunicationState.NORMAL;

    // 回调执行器
    IBProtocolResponse defaultResponse;

    // 协议控制器
    IProtocolManager manager;

    // 指令响应回调器
    Map<String, ProtocolEntity> responseMap;

    // 回调事件异步线程
    private ADReadAsynThread readAsynThread;

    public SealClientEventListener(IBProtocolResponse defaultResponse, IProtocolManager manager) {
        this.defaultResponse = defaultResponse;
        this.manager = manager;
        this.responseMap = new HashMap<>();
        this.readAsynThread = new ADReadAsynThread(this);
        this.readAsynThread.switchOn();
        this.readAsynThread.start();
    }

    /**
     * @param bytes
     * @description: 连接监听
     */
    @Override
    public void onClientEvent(byte[] bytes) throws MeterException {
        boolean other = this.defaultResponse != null && this.defaultResponse instanceof IOProtocolResponse;
        if (other) {
            ((IOProtocolResponse) this.defaultResponse).target(bytes);
        } else {
            boolean debug = this.defaultResponse != null && this.defaultResponse instanceof IDebugProtocolResponse;
            if (bytes != null && bytes.length > 0) {
                // System.out.println("源码： " + ByteUtils.toHexString(bytes));
                if (debug) ((IDebugProtocolResponse) this.defaultResponse).source(bytes);
                byte[] decode = CodeUtils.decode(bytes);
                // System.out.println("解码： " + ByteUtils.toHexString(decode));
                if (debug) ((IDebugProtocolResponse) this.defaultResponse).target(decode);
                // 添加缓冲区
                this.addBuffer(decode);
                // ADCache.getInstance().addBuffer(decode);
            }
        }
    }

    /**
     * @description: 默认回调器
     **/
    @Override
    public IBProtocolResponse defaultResponse() {
        return this.defaultResponse;
    }

    /**
     * @param buffer
     * @description: 执行解析发送
     */
    @Override
    public byte[] execute(byte[] buffer) {
        // System.out.println("Read Thread: " + ByteUtils.toHexString(buffer));
        if (buffer[0] != 0X02) {
            boolean has_prefix = false;
            for (int i = 0; i < buffer.length; i++) {
                if (buffer[i] == 0X02) {
                    has_prefix = true;
                    buffer = Arrays.copyOfRange(buffer, i, buffer.length);
                    break;
                }
            }
            if (!has_prefix)
                return new byte[0];
        }

        List<byte[]> bytes = new ArrayList<>();
        int index = 0;
        for (int i = 0; i < buffer.length; i++) {
            if (buffer[i] == 0X03) {
                if (i + 1 != buffer.length && buffer[i + 1] == 0X02) {
                    bytes.add(Arrays.copyOfRange(buffer, index, i + 1));
                    index = i + 1;
                    continue;
                }

                if (i + 1 == buffer.length) {
                    byte[] copy = Arrays.copyOfRange(buffer, index, i + 1);
                    if (Integer.parseInt(ByteUtils.toHexString(Arrays.copyOfRange(copy, 1, 2)), 16) + 5 == copy.length) {
                        bytes.add(copy);
                        index = i + 1;
                        continue;
                    }
                }
            }
        }

        try {
            if (!bytes.isEmpty()) {
                for (byte[] aByte : bytes) {
                    if (GlobalPara.getInstance().isRunLog())
                        System.out.println("解析：" + ByteUtils.toHexString(aByte));
                    // 内容解析
                    ADPackDisassemblyResponse response = (ADPackDisassemblyResponse) this.manager.execute(new ADPackDisassemblyRequest(aByte));
                    ProtocolEntity protocolEntity = this.responseMap.remove(response.KEY());
                    if (protocolEntity != null && protocolEntity.getResponse() instanceof IProtocolResponse) {
                        ((IProtocolResponse) protocolEntity.getResponse()).target(response);
                    } else if (this.defaultResponse != null && this.defaultResponse instanceof IProtocolResponse) {
                        ((IProtocolResponse) this.defaultResponse).target(response);
                    } else {
                        throw new MeterException(MeterStateEnum.AD_响应通道对象异常);
                    }
                }
            }
        } catch (MeterException e) {
            e.printStackTrace();
            JLog.error("AD Parse error：{}", e.getMessage(), e);
            if (this.defaultResponse != null) this.defaultResponse.exp(e);
        } catch (Exception e) {
            e.printStackTrace();
            JLog.error("AD Parse Exception：{}", e.getMessage(), e);
            if (this.defaultResponse != null)
                this.defaultResponse.exp(new MeterException(e, MeterStateEnum.其他未知异常));
        }
        return Arrays.copyOfRange(buffer, index, buffer.length);
    }

    /**
     * @param bytes
     * @description: 添加缓冲区
     */
    @Override
    public void addBuffer(byte[] bytes) {
        this.readAsynThread.addBuffer(bytes);
    }

    /**
     * @description: 缓冲区清空
     **/
    @Override
    public void clear() {
        this.readAsynThread.clear();
    }

    /**
     * @description: 过期检查
     * @date: 2024-08-16 14:04:19
     **/
    @Override
    public synchronized void expireCheck() {
        try {
            if (this.responseMap != null && !this.responseMap.isEmpty()) {
                Iterator<Map.Entry<String, ProtocolEntity>> iterator = this.responseMap.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry<String, ProtocolEntity> entry = iterator.next();
                    if (entry.getValue() == null || entry.getValue().expire(10000))
                        iterator.remove();
                }
            }
        } catch (Exception e) {
        }
    }

    /**
     * @param e
     * @description: 连接异常监听
     */
    @Override
    public void onClientEventExp(MeterException e) {
        if (this.defaultResponse != null)
            this.defaultResponse.exp(e);
    }

    /**
     * @param key
     * @param response
     * @description: 发送命令添加回调
     */
    @Override
    public synchronized void addProtocolResponse(String key, ProtocolEntity response) {
        this.responseMap.put(key, response);
    }

    /**
     * @description: 监听器停止
     **/
    @Override
    public void stop() {
        this.readAsynThread.giveOver();
    }

    /**
     * @description: 修改通讯状态
     **/
    public synchronized void setCommunication(CommunicationState communication) {
        this.communication = communication;
        this.readAsynThread.switchOn();
        try {
            Thread.sleep(500);
        } catch (InterruptedException e) {
        }
        this.readAsynThread.switchOn();
    }

    /**
     * @description: 获取当前通讯协议状态
     **/
    @Override
    public CommunicationState getCommunication() {
        return this.communication;
    }
}
