package com.jhscale.meter.tool.barcode;

import com.jhscale.common.model.simple.JSONModel;
import com.jhscale.meter.em.PayCode;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.protocol.constant.TMS;
import com.jhscale.meter.protocol.model.BarCodeItemResult;
import com.jhscale.meter.protocol.model.BarCodeItemSearch;
import com.jhscale.meter.protocol.model.GlobalPara;
import com.jhscale.meter.tool.barcode.em.Commodity;
import com.jhscale.meter.tool.barcode.entity.*;
import com.jhscale.meter.tool.barcode.loader.BarCodeLoader;
import com.jhscale.meter.tool.barcode.loader.CashierCodeLoader;
import com.jhscale.meter.tool.barcode.loader.JKYPayCodeLoader;
import com.jhscale.meter.tool.barcode.loader.PLUSpecsLoader;
import com.jhscale.meter.tool.barcode.loader.rule.*;
import org.apache.commons.lang3.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

import static com.jhscale.meter.utils.Constant.TEMP_PCS;
import static com.jhscale.meter.utils.Constant.TEMP_Weight;

/**
 * @author lie_w
 * @title: BarCodeTool
 * @projectName meter-jar
 * @description: 条码工厂
 * @date 2022/8/48:21
 */
public class BarCodeFactory extends JSONModel {

    // 吉卡云条码规则
    private JKYPayCodeLoader jkyPayCodeLoader;

    // 集中收银规则
    private CashierCode cashierCode;

    private BarCodeFactory() {
    }

    private static class SingletonFactory {
        private static BarCodeFactory INSTANCE = new BarCodeFactory();
    }


    public static BarCodeFactory getInstance() {
        return SingletonFactory.INSTANCE;
    }

    private static Map<Integer, CashierCode> cashierCodeMap;

    static {
        // 加载收银条码规则
        cashierCode();
    }

    /**
     * @description: 加载收银条码 规则信息
     **/
    private static Map<Integer, CashierCode> cashierCode() {
        cashierCodeMap = new HashMap<>();

        CashierCode_13_1 cashierCode_13_1 = new CashierCode_13_1();
        cashierCodeMap.put(cashierCode_13_1.sn(), cashierCode_13_1);
        CashierCode_13_2 cashierCode_13_2 = new CashierCode_13_2();
        cashierCodeMap.put(cashierCode_13_2.sn(), cashierCode_13_2);
        CashierCode_13_3 cashierCode_13_3 = new CashierCode_13_3();
        cashierCodeMap.put(cashierCode_13_3.sn(), cashierCode_13_3);
        CashierCode_13_4 cashierCode_13_4 = new CashierCode_13_4();
        cashierCodeMap.put(cashierCode_13_4.sn(), cashierCode_13_4);
        CashierCode_13_5 cashierCode_13_5 = new CashierCode_13_5();
        cashierCodeMap.put(cashierCode_13_5.sn(), cashierCode_13_5);

        CashierCode_18_1 cashierCode_18_1 = new CashierCode_18_1();
        cashierCodeMap.put(cashierCode_18_1.sn(), cashierCode_18_1);
        CashierCode_18_2 cashierCode_18_2 = new CashierCode_18_2();
        cashierCodeMap.put(cashierCode_18_2.sn(), cashierCode_18_2);
        CashierCode_18_3 cashierCode_18_3 = new CashierCode_18_3();
        cashierCodeMap.put(cashierCode_18_3.sn(), cashierCode_18_3);
        CashierCode_18_4 cashierCode_18_4 = new CashierCode_18_4();
        cashierCodeMap.put(cashierCode_18_4.sn(), cashierCode_18_4);
        CashierCode_18_5 cashierCode_18_5 = new CashierCode_18_5();
        cashierCodeMap.put(cashierCode_18_5.sn(), cashierCode_18_5);

        return cashierCodeMap;
    }

    /**
     * @description: 添加加载器
     **/
    public BarCodeFactory build(BarCodeLoader... loaders) throws MeterException {
        if (loaders != null && loaders.length > 0) {
            for (BarCodeLoader loader : loaders) {
                if (loader instanceof JKYPayCodeLoader) {
                    this.jkyPayCodeLoader = (JKYPayCodeLoader) loader;
                } else if (loader instanceof CashierCodeLoader) {
                    this.cashierCode = cashierCodeMap.get(((CashierCodeLoader) loader).getRule().serialNo());
                    if (this.cashierCode == null)
                        throw new MeterException(MeterStateEnum.加载集中收银失败);
                    this.cashierCode.setRule(((CashierCodeLoader) loader).getRule());
                }
            }
        }
        return this;
    }

    // 微信支付码正则
    private static final String WX_PAY_REGULAR = "^1[0-5]\\d{16}$";
    // 支付宝支付码正则
    private static final String ALI_PAY_REGULAR = "^(?:2[5-9]|30)\\d{14,18}$";

    /**
     * @description: 条码识别
     **/
    public PayCode payCode(String code) {
        if (Pattern.matches(WX_PAY_REGULAR, code)) {
            return PayCode.WECHAT_PAY;
        } else if (Pattern.matches(ALI_PAY_REGULAR, code)) {
            return PayCode.ALI_PAY;
        } else if (Objects.nonNull(this.jkyPayCodeLoader) && Pattern.matches(this.jkyPayCodeLoader.regular(), code)) {
            return PayCode.JK_PAY;
        }
        return PayCode.UNION_PAY;
    }

    /**
     * @description: 商品条码 识别商品
     **/
    @Deprecated
    public CommodityResult commodityParse(String code) {
        if (StringUtils.isNotBlank(code)
                && Objects.nonNull(this.cashierCode)
                && this.cashierCode.checkLength(code)
                && this.cashierCode.checkPart1(code)) {
            BarCodeInfo barCodeInfo = new BarCodeInfo();
            barCodeInfo.setPart2(this.cashierCode.getRule().getPart2());
            this.cashierCode.parse(code, barCodeInfo);
            return new CommodityResult(Commodity.收银条码.getType(), code, barCodeInfo);
        }
        return new CommodityResult(Commodity.普通条码.getType(), code, null);
    }

    /**
     * @description: 商品条码 识别商品后 根据基本信息 查询 计重计件类型 以及售价 完善 完整收银条码信息
     **/
    @Deprecated
    public CashierResult commodityComplete(BarCodeInfo barCodeInfo, PLUSpecsParam pluSpecsParam) {
        if (barCodeInfo != null && Objects.nonNull(this.cashierCode)) {
            if (pluSpecsParam != null) {
                this.cashierCode.complete(barCodeInfo, pluSpecsParam.getUnit(), pluSpecsParam.getPrice());
            }
            return new CashierResult(Commodity.收银条码.getType(),
                    barCodeInfo,
                    barCodeInfo.pluNumberNoConstant(),
                    pluSpecsParam,
                    barCodeInfo.getAmount(), barCodeInfo.getWeight(), barCodeInfo.getPrice(), barCodeInfo.getTotal());
        }
        return new CashierResult(Commodity.普通条码.getType());
    }

    /**
     * @description: 条码完整数据内容解析，需要提供查询接口 获取 商品 计件计重类型 外加 单价
     **/
    @Deprecated
    public CashierResult commodity(String code, PLUSpecsLoader pluSpecsLoader) {
        CommodityResult commodityResult = this.commodityParse(code);
        if (commodityResult.getType() == Commodity.收银条码.getType()) {
            if (pluSpecsLoader != null) {
                return this.commodityComplete(commodityResult.getBarCodeInfo(), pluSpecsLoader.queryPLUSpecsByNumberOrNoOrConstant(commodityResult.getBarCodeInfo().pluNumberNoConstant()));
            }
            return new CashierResult(Commodity.收银条码.getType(), code, commodityResult.getBarCodeInfo());
        } else {
            return new CashierResult(Commodity.普通条码.getType(), code);
        }
    }

    /**
     * @description: 商品收银条码组装
     **/
    @Deprecated
    public String commodityAssemble(PLUSpecsItem item) throws MeterException {
        if (Objects.nonNull(item) && Objects.nonNull(this.cashierCode)) {
            return new StringBuffer()
                    .append(this.cashierCode.getRule().getPart1())
                    .append(this.cashierCode.addPart2(item))
                    .append(this.cashierCode.addPart3(item))
                    .append(this.cashierCode.addPart4(item))
                    .append(this.cashierCode.addVerify())
                    .toString();
        }
        throw new MeterException(MeterStateEnum.加解密模块未初始化);
    }

    /**
     * @description: EAN 13 2+5+5+1
     **/
    @Deprecated
    public BarCodeFactory cashierCodeLoader_13_1(Rule rule) throws MeterException {
        rule.setEan(13);
        rule.setNo(1);
        return this.build(new CashierCodeLoader(rule));
    }

    /**
     * @description: EAN 18 2+5+5+5+1
     **/
    @Deprecated
    public BarCodeFactory cashierCodeLoader_18_1(Rule rule) throws MeterException {
        rule.setEan(18);
        rule.setNo(1);
        return this.build(new CashierCodeLoader(rule));
    }

    /**
     * @description: 条码完整数据内容解析，需要提供查询接口 获取 商品 计件计重类型 外加 单价
     **/
    public CashierResult Commodity_Format(String code, PLUSpecsLoader pluSpecsLoader) throws MeterException {
        if (StringUtils.isBlank(code)) return new CashierResult(Commodity.普通条码.getType(), code);

        BarCodeItemResult barCodeItemResult = BarCodeItemSearch.Calulate(code);

        if (barCodeItemResult == null || !barCodeItemResult.isOK())
            return new CashierResult(Commodity.普通条码.getType(), code);

        Integer pluNo = barCodeItemResult.getNumber();
        Integer number = barCodeItemResult.getItemcode();

        PLUNumberNoConstant pluNumberNoConstant = null;
        if (Objects.nonNull(pluNo)) {
            pluNumberNoConstant = new PLUNumberNoConstant(null, pluNo, null);
        } else if (Objects.nonNull(number)) {
            if (number == 0) {
                int Scanner_0Note_Deal = GlobalPara.getInstance().getTMS(TMS.Scanner_0Note_Deal);
                if (Scanner_0Note_Deal == 1) {
                    pluNumberNoConstant = new PLUNumberNoConstant(null, TEMP_Weight, null);
                } else if (Scanner_0Note_Deal == 2) {
                    pluNumberNoConstant = new PLUNumberNoConstant(null, TEMP_PCS, null);
                }
            } else {
                pluNumberNoConstant = new PLUNumberNoConstant(number, null, null);
            }
        }

        if (Objects.isNull(pluNumberNoConstant)) return new CashierResult(Commodity.普通条码.getType(), code);

        if (Objects.isNull(pluSpecsLoader))
            return new CashierResult(Commodity.收银条码.getType(), code, barCodeItemResult, pluNumberNoConstant);

        PLUSpecsParam pluSpecsParam = pluSpecsLoader.queryPLUSpecsByNumberOrNoOrConstant(pluNumberNoConstant);
        if (Objects.isNull(pluSpecsParam)) {
            return new CashierResult(Commodity.收银条码.getType(), code, barCodeItemResult, pluNumberNoConstant);
        } else {
            // if ((Objects.isNull(pluSpecsParam.getPrice()) || BigDecimalUtils.compareZeroValue(pluSpecsParam.getPrice()))
            //         &&
            //         (Objects.isNull(barCodeItemResult.getAmount()) || Objects.isNull(barCodeItemResult.getPrice()))) {
            //     throw new MeterException(MeterStateEnum.条码格式13位且本机商品价格为0不合法);
            // }
            return new CashierResult(Commodity.收银条码.getType(), code, barCodeItemResult, pluNumberNoConstant, pluSpecsParam);
        }
    }
}
