package com.jhscale.meter.utils;

import com.jhscale.meter.model.device.TradeConvert;
import com.jhscale.meter.protocol.model.GlobalPara;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author lie_w
 * @title: BigDecimalUtils
 * @projectName meter-jar
 * @description: TODO
 * @date 2022/8/51:51
 */
public class BigDecimalUtils {

    private static final BigDecimal BASIC = new BigDecimal(10);
    private static Map<String, BigDecimal> unitConvert = new HashMap<>();

    static {
        unitConvert.put("重量", new BigDecimal("1000"));
        unitConvert.put("g", new BigDecimal("1"));
        unitConvert.put("kg", new BigDecimal("1000"));
        unitConvert.put("ton", new BigDecimal("1000000"));
        unitConvert.put("t", new BigDecimal("1000000"));
        unitConvert.put("lb", new BigDecimal("453.59237"));
        unitConvert.put("b", new BigDecimal("453.59237"));
        unitConvert.put("500g", new BigDecimal("500"));
        unitConvert.put("100g", new BigDecimal("100"));
        unitConvert.put("1/4lb", new BigDecimal("113.39809"));
        unitConvert.put("1/4b", new BigDecimal("113.39809"));
    }

    private BigDecimalUtils() {
    }

    /**
     * @description: 基数转换
     **/
    public static BigDecimal cardinality(int cardinality) {
        BigDecimal val = new BigDecimal(1);
        for (int i = 0; i < cardinality; i++) {
            val = val.multiply(BASIC);
        }
        return val;
    }

    /**
     * @description: 添加自定义单位转换(已存在的单位不允许修改)
     **/
    public static boolean addUnitConvert(String unit, BigDecimal val) {
        BigDecimal bigDecimal = unitConvert.get(unit);
        if (bigDecimal == null) {
            unitConvert.put(unit, val);
            return true;
        }
        return false;
    }

    /**
     * @param weight     重量数值
     * @param weightUnit 重量单位
     * @param price      单价数值
     * @param priceUnit  单价单位
     * @description: 单价交易小计信息转换（转换单价）[优先考虑 价格精度丢失 但是库存精度不会丢失]
     **/
    public static TradeConvert priceItem(BigDecimal weight, String weightUnit, BigDecimal price, String priceUnit, int scale, int round) {
        if (weightUnit.equalsIgnoreCase(priceUnit)) {
            return new TradeConvert(price, weight, price.multiply(weight).setScale(scale, round));
        } else {
            BigDecimal weightBasic = unitConvert.get(weightUnit);
            BigDecimal priceBasic = unitConvert.get(priceUnit);
            price = price.multiply(weightBasic).divide(priceBasic, 10, BigDecimal.ROUND_HALF_UP);
            return new TradeConvert(price.setScale(scale, round), weight, price.multiply(weight).setScale(scale, round));
        }
    }

    /**
     * @param weight     重量数值
     * @param weightUnit 重量单位
     * @param price      单价数值
     * @param priceUnit  单价单位
     * @description: 单价交易小计信息转换（重量转换）[重量精度丢失]
     **/
    public static TradeConvert weightItem(BigDecimal weight, String weightUnit, BigDecimal price, String priceUnit, int scale, int round) {
        if (weightUnit.equalsIgnoreCase(priceUnit)) {
            return new TradeConvert(price, weight, price.multiply(weight).setScale(scale, round));
        } else {
            BigDecimal weightBasic = unitConvert.get(weightUnit);
            BigDecimal priceBasic = unitConvert.get(priceUnit);
            weight = weight.multiply(weightBasic).divide(priceBasic, 10, BigDecimal.ROUND_HALF_UP);
            return new TradeConvert(price, weight.setScale(scale, round), price.multiply(weight).setScale(scale, round));
        }
    }

    /**
     * @description: 金额转字符串
     **/
    public static String bigDecimalToString(BigDecimal val, Integer scale, Integer round, Integer defaultScale, Integer defaultRound) {
        if (Objects.isNull(scale) && Objects.nonNull(defaultScale)) scale = defaultScale;
        if (Objects.isNull(scale)) scale = 2;
        if (Objects.isNull(round) && Objects.nonNull(defaultRound)) round = defaultRound;
        if (Objects.isNull(round)) round = BigDecimal.ROUND_HALF_UP;
        return (Objects.nonNull(val) ? val : new BigDecimal(0)).setScale(scale, round).toString();
    }

    /**
     * @description: 金额转字符串
     **/
    public static String bigDecimalToString(BigDecimal val, Integer scale) {
        return GlobalPara.getInstance().GetStringTMS(new BigDecimal(bigDecimalToString(val, scale, null, null, null)));
    }

    public static String bigDecimalToString(BigDecimal val, Integer scale, Integer defaultScale) {
        return GlobalPara.getInstance().GetStringTMS(new BigDecimal(bigDecimalToString(val, scale, null, defaultScale, null)));
    }


//    public static void main(String[] args) {
//        BigDecimal weight = new BigDecimal("4.5");
//        String weightUnit = "kg";
//        BigDecimal price = new BigDecimal("5.3");
//        String priceUnit = "lb";
//
//        System.out.println(priceItem(weight, weightUnit, price, priceUnit, 4, BigDecimal.ROUND_HALF_UP).toJSON());
//        System.out.println(weightItem(weight, weightUnit, price, priceUnit, 4, BigDecimal.ROUND_HALF_UP).toJSON());
//    }
}
