package com.jhscale.meter.utils;

import com.intel.bluetooth.*;
import com.jhscale.meter.model.device.BlueDevice;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.io.BluetoothPort;
import com.jhscale.meter.log.JLog;
import org.apache.commons.lang3.StringUtils;

import javax.bluetooth.LocalDevice;
import javax.bluetooth.RemoteDevice;
import javax.microedition.io.Connector;
import java.io.IOException;
import java.lang.reflect.Field;

/**
 * @author Loveven
 * @title: BluCatUtil
 * @projectName master-jar
 * @description: 蓝牙检查工具
 * @date 2021-06-0420:18
 */
public class BluecoveUtils {

    private BluecoveUtils() {
    }

    /**
     * @description: 蓝牙驱动检查
     **/
    public static void doctorDevice() throws MeterException {
        if (System.getProperty("os.name").contains("Linux")) {
            JLog.info("{} os device check info [ {} ]", BluetoothPort.TAG, "Is libbluetooth3 and libbluetooth-dev installed?");
            JLog.info("{} os device check info [ {} ]", BluetoothPort.TAG, "run: sudo apt-get install libbluetooth3 libbluetooth-dev");
        }

        if (!LocalDevice.isPowerOn()) {
            throw new MeterException(MeterStateEnum.未发现蓝牙驱动设备);
        }

        JLog.info("{} BluCatUtil info [ {} ]", BluetoothPort.TAG, "BlueCoveState");

        try {
            JLog.debug("{} BluCatUtil ThreadBluetoothStackID：[ {} ]", BluetoothPort.TAG, BlueCoveImpl.getThreadBluetoothStackID());
            JLog.debug("{} BluCatUtil CurrentThreadBluetoothStackID：[ {} ]", BluetoothPort.TAG, BlueCoveImpl.getCurrentThreadBluetoothStackID());
            JLog.debug("{} BluCatUtil LocalDevicesID：[ {} ]", BluetoothPort.TAG, BlueCoveImpl.getLocalDevicesID());
        } catch (Exception e) {
            JLog.error("{} BluCatUtil Bluthtooth info error：[ {} ]", BluetoothPort.TAG, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.检查蓝牙信息失败);
        }

        JLog.info("{} BluCatUtil Bluthtooth info [ {} ]", BluetoothPort.TAG, "BlueCoveConfigProperties");

        String fieldName = null;
        try {
            for (Field f : BlueCoveConfigProperties.class.getDeclaredFields()) {
                if (f.getName().startsWith("PROPERTY")) {
                    fieldName = f.getName();
                    String PROPERTY = String.valueOf(LocalDevice.getProperty(String.valueOf(f.get(null))));
                    JLog.debug("{} BluCatUtil PROPERTY:[ {} ]-[ {} ]", BluetoothPort.TAG, fieldName, PROPERTY);
                }
            }
        } catch (Exception e) {
            JLog.error("{} BluCatUtil PROPERTY [ {} ] error：[ {} ]", BluetoothPort.TAG, fieldName, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.检查蓝牙PROPERTY信息失败);
        }

        JLog.info("{} BluCatUtil info [ {} ]", BluetoothPort.TAG, "LocalDeviceProperties");

        try {
            String[] deviceprops = {
                    "bluetooth.api.version",
                    "bluetooth.master.switch",
                    "bluetooth.sd.attr.retrievable.max",
                    "bluetooth.connected.devices.max",
                    "bluetooth.l2cap.receiveMTU.max",
                    "bluetooth.sd.trans.max",
                    "bluetooth.connected.inquiry.scan",
                    "bluetooth.connected.page.scan",
                    "bluetooth.connected.inquiry",
                    "bluetooth.connected.page"
            };
            for (String prop : deviceprops) {
                String property = LocalDevice.getProperty(prop);
                JLog.debug("{} BluCatUtil deviceprops：[ {} ]-[ {} ]", BluetoothPort.TAG, prop, property);
            }
            for (Field f : BlueCoveLocalDeviceProperties.class.getDeclaredFields()) {
                if (f.getName().startsWith("LOCAL_DEVICE")) {
                    fieldName = f.getName();
                    String LOCAL_DEVICE = String.valueOf(LocalDevice.getProperty(String.valueOf(f.get(null))));
                    JLog.debug("{} BluCatUtil LOCAL_DEVICE:[ {} ]-[ {} ]", BluetoothPort.TAG, fieldName, LOCAL_DEVICE);
                }
            }
        } catch (Exception e) {
            JLog.error("{} BluCatUtil LOCAL_DEVICE [ {} ] error：[ {} ]", BluetoothPort.TAG, fieldName, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.检查蓝牙LOCAL_DEVICE信息失败);
        }

        JLog.info("{} BluCatUtil info [ {} ]", BluetoothPort.TAG, "LocalDeviceFeatures");

        try {
            for (Field f : BluetoothStack.class.getDeclaredFields()) {

                if (f.getName().startsWith("FEATURE")) {
                    fieldName = f.getName();
                    String FEATURE = String.valueOf(BlueCoveImpl.instance().getLocalDeviceFeature((f.getInt(null))));
                    JLog.debug("{} BluCatUtil FEATURE:[ {} ]-[ {} ]", BluetoothPort.TAG, fieldName, FEATURE);
                }
            }
        } catch (Exception e) {
            JLog.error("{} BluCatUtil FEATURE [ {} ] error：[ {} ]", BluetoothPort.TAG, fieldName, e.getMessage(), e);
            throw new MeterException(MeterStateEnum.检查蓝牙FEATURE信息失败);
        }
        JLog.info("{} BluCatUtil device Success.", BluetoothPort.TAG);
    }

    /**
     * @description: 获取蓝牙名称
     **/
    public static String deviceName(RemoteDevice device) {
        String name = null;
        int count = 0;
        do {
            try {
                name = device.getFriendlyName(false);
            } catch (IOException e) {
                JLog.error("{} deviceName 地址:{} 获取名称异常：{}", BluetoothPort.TAG, device.getBluetoothAddress(), e.getMessage(), e);
            }
            count++;
        } while (StringUtils.isBlank(name) && count < 3);
        return clean(name);
    }

    /**
     * @description: 获取设备信息
     **/
    public static BlueDevice deviceInfo(RemoteDevice device) {
        String address = device.getBluetoothAddress();
        String name = deviceName(device);
        BlueDevice<RemoteDevice> blueDevice = new BlueDevice<RemoteDevice>(address, device, name, device.isTrustedDevice(), device.isEncrypted());

        String rssi = "NA";
        String toret = "";
        toret += clean(address) + ", " + "\"" + clean(name) + "\", " + "Trusted:"
                + device.isTrustedDevice() + ", " + "Encrypted:" + device.isEncrypted();
        if (BlucatState.rssi) {
            try {
                rssi = String.valueOf(RemoteDeviceHelper.readRSSI(device));
            } catch (Throwable e) {
                String url = "btl2cap://" + device.getBluetoothAddress() + ":1";
                try {
                    BlucatState.connection = Connector.open(url, Connector.READ_WRITE, true);
                    rssi = String.valueOf(RemoteDeviceHelper.readRSSI(device));
                    BlucatState.connection.close();
                } catch (IOException e1) {
                }
            }
            toret += ", " + rssi;
        }
        blueDevice.setRemark(toret);
        return blueDevice;
    }

    /**
     * @description: 去空处理
     **/
    public static String clean(String str) {
        return StringUtils.isNotBlank(str) ?
                str.replace("\"", "''")
                        .replace("\n", " ") : str;
    }
}