package com.jhscale.meter.utils;

import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import gnu.io.*;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.TooManyListenersException;

/**
 * @author lie_w
 * @title: SerialTool
 * @projectName jhscale-agreement
 * @description: 串口工具
 * https://www.cnblogs.com/Dreamer-1/p/5523046.html
 * @date 2021-01-0715:48
 */
public class SerialTool {

    private static SerialTool serialTool = null;

    static {
        //在该类被ClassLoader加载时就初始化一个SerialTool对象
        if (serialTool == null) serialTool = new SerialTool();
    }

    /**
     * @description: 获取提供服务的SerialTool对象
     **/
    public static SerialTool getSerialTool() {
        if (serialTool == null)
            serialTool = new SerialTool();
        return serialTool;
    }

    /**
     * @description: 查找所有可用端口
     **/
    public static final List<String> findPort() {
        //获得当前所有可用串口
        Enumeration<CommPortIdentifier> portList = CommPortIdentifier.getPortIdentifiers();
        List<String> portNameList = new ArrayList<>();

        //将可用串口名添加到List并返回该List
        while (portList.hasMoreElements()) {
            String portName = portList.nextElement().getName();
            portNameList.add(portName);
        }
        return portNameList;
    }

    /**
     * @param portName 端口名称
     * @param baudrate 波特率
     * @return 串口对象
     * @description: 打开串口
     **/
    public static final SerialPort openPort(String portName, int baudrate) throws MeterException {
        try {
            //通过端口名识别端口
            CommPortIdentifier portIdentifier = CommPortIdentifier.getPortIdentifier(portName);

            //打开端口，并给端口名字和一个timeout（打开操作的超时时间）
            CommPort commPort = portIdentifier.open(portName, 2000);

            if (commPort instanceof SerialPort) {
                SerialPort serialPort = (SerialPort) commPort;
                //设置一下串口的波特率等参数
                serialPort.setSerialPortParams(baudrate, SerialPort.DATABITS_8, SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);
                return serialPort;
            } else {
                throw new MeterException(MeterStateEnum.端口指向设备不是串口类型);
            }
        } catch (NoSuchPortException e) {
            e.printStackTrace();
            throw new MeterException(MeterStateEnum.没有该端口对应的串口设备);
        } catch (PortInUseException e) {
            e.printStackTrace();
            throw new MeterException(MeterStateEnum.端口已被占用);
        } catch (UnsupportedCommOperationException e) {
            throw new MeterException(MeterStateEnum.设置串口参数失败);
        }
    }

    /**
     * @description: 待关闭的串口对象
     **/
    public static void closePort(SerialPort serialPort) {
        if (serialPort != null) {
            serialPort.close();
            serialPort = null;
        }
    }

    /**
     * @param serialPort 串口对象
     * @param order      待发送数据
     * @description: 往串口发送数据
     */
    public static void sendToPort(SerialPort serialPort, byte[] order) throws MeterException {
        OutputStream out = null;
        try {
            out = serialPort.getOutputStream();
            out.write(order);
            out.flush();
        } catch (IOException e) {
            throw new MeterException(MeterStateEnum.向串口发送数据失败);
        } finally {
            try {
                if (out != null) {
                    out.close();
                    out = null;
                }
            } catch (IOException e) {
                throw new MeterException(MeterStateEnum.关闭串口对象的输出流出错);
            }
        }
    }

    /**
     * 从串口读取数据
     *
     * @param serialPort 当前已建立连接的SerialPort对象
     * @return 读取到的数据
     */
    public static byte[] readFromPort(SerialPort serialPort) throws MeterException {
        InputStream in = null;
        byte[] bytes = null;
        try {
            in = serialPort.getInputStream();
            int bufflenth = in.available();        //获取buffer里的数据长度

            while (bufflenth != 0) {
                bytes = new byte[bufflenth];    //初始化byte数组为buffer中数据的长度
                in.read(bytes);
                bufflenth = in.available();
            }
        } catch (IOException e) {
            throw new MeterException(MeterStateEnum.从串口读取数据时出错);
        } finally {
            try {
                if (in != null) {
                    in.close();
                    in = null;
                }
            } catch (IOException e) {
                throw new MeterException(MeterStateEnum.关闭串口对象输入流出错);
            }
        }
        return bytes;
    }

    /**
     * 添加监听器
     *
     * @param port     串口对象
     * @param listener 串口监听器
     */
    public static void addListener(SerialPort port, SerialPortEventListener listener) throws MeterException {
        try {
            //给串口添加监听器
            port.addEventListener(listener);
            //设置当有数据到达时唤醒监听接收线程
            port.notifyOnDataAvailable(true);
            //设置当通信中断时唤醒中断线程
            port.notifyOnBreakInterrupt(true);
        } catch (TooManyListenersException e) {
            e.printStackTrace();
            throw new MeterException(MeterStateEnum.监听类对象过多);
        }
    }
}
