package com.jhscale.meter.wifi.entity;

import com.jhscale.common.utils.ASCIIUtils;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.Constant;
import com.jhscale.meter.wifi.model.PackageMark;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Date;
import java.util.Objects;

/**
 * @author lie_w
 * @title: CommandResponse
 * @projectName jhscale-agreement
 * @description: 响应对象
 * @date 2020/6/229:54
 */
public class CommandResponse extends JHObject {

    // 整数包Mark
    private PackageMark packageMark;

    // NID
    private String nid;

    // 内容长度
    private int contentLength;

    // 字符集
    private String encoding;

    private Charset charset;

    // 应答内容段
    private StringBuffer responseContent;

    /**
     * @description: 开始应答拼装
     **/
    public CommandResponse start() {
        this.responseContent = new StringBuffer();
        return this;
    }

    /**
     * @description: 结束拼装
     **/
    public String over() {
        return this.getResponseContent().toString();
    }

    /**
     * @description: 添加ASCII字符
     **/
    public CommandResponse appendAscii(String ascii) {
        this.responseContent.append(ASCIIUtils.convertStringToHex(ascii));
        return this;
    }

    /**
     * 添加HEX
     *
     * @param hex
     * @return
     */
    public CommandResponse appendHex(String hex) {
        this.responseContent.append(hex);
        return this;
    }

    /**
     * 添加不定长文本
     *
     * @param part
     * @return
     */
    public CommandResponse appendText(String part) {
        if (StringUtils.isNotBlank(part))
            this.responseContent.append(ByteUtils.toHexString(part.getBytes(charset)));
        this.responseContent.append("00");
        return this;
    }

    /**
     * 添加不定长文本
     *
     * @param part
     * @return
     */
    public CommandResponse appendTextLine(String part) {
        if (StringUtils.isNotBlank(part)) {
            if (part.contains(System.lineSeparator())) {
                String[] parts = part.split(System.lineSeparator());
                for (String text : parts) {
                    this.responseContent.append(ByteUtils.toHexString(ByteUtils.byteMerger(text.getBytes(charset), new byte[]{0x0b})));
                }
            } else if (part.contains("\n")) {
                String[] parts = part.split("\\n");
                for (String text : parts) {
                    this.responseContent.append(ByteUtils.toHexString(ByteUtils.byteMerger(text.getBytes(charset), new byte[]{0x0b})));
                }
            } else {
                this.responseContent.append(ByteUtils.toHexString(part.getBytes(charset)));
            }
        }
        this.responseContent.append("00");
        return this;
    }

    /**
     * 添加不定长文本
     *
     * @param part
     * @return
     */
    public CommandResponse append(String part) {
        if (StringUtils.isNotBlank(part))
            this.responseContent.append(ByteUtils.toHexString(part.getBytes(charset)));
        return this;
    }

    /**
     * 添加1字节
     *
     * @param character
     * @return
     */
    public CommandResponse append(char character) {
        return appendBytes(character, 2);
    }

    /**
     * 添加1字节Integer
     *
     * @param integer
     * @return
     */
    public CommandResponse append1Bytes(int integer) {
        return appendBytes(integer, 2);
    }

    /**
     * 添加Integer
     *
     * @param integer
     * @return
     */
    public CommandResponse append(int integer) {
        this.responseContent.append(ByteUtils.convertHex(Long.toHexString(integer)));
        return this;
    }

    /**
     * 添加2字节Integer
     *
     * @param integer
     * @return
     */
    public CommandResponse append2Bytes(int integer) {
        return this.appendBytes(integer, 4);
    }

    /**
     * 添加4字节Long
     *
     * @param data
     * @return
     */
    public CommandResponse append4Bytes(long data) {
        return this.appendBytes(data, 8);
    }

    /**
     * 添加4字节BigDecimal
     *
     * @param bigDecimal
     * @return
     */
    public CommandResponse append(BigDecimal bigDecimal) {
        this.responseContent.append(ByteUtils.completeBytes(Long.toHexString(ByteUtils.toSPF(bigDecimal)), 8));
        return this;
    }

    /**
     * 添加2字节Short
     *
     * @param tmp
     * @return
     */
    public CommandResponse append(short tmp) {
        return this.appendBytes(tmp, 4);
    }

    /**
     * 添加4字节日期
     *
     * @param date
     * @return
     */
    public CommandResponse append(Date date) {
        return this.append(date, null);
    }

    /**
     * 添加4字节日期
     *
     * @param date
     * @return
     */
    public CommandResponse append(Date date, Integer timeZone) {
        if (Objects.isNull(timeZone)) timeZone = 8;
        LocalDateTime start = ByteUtils.getStartTime();
        LocalDateTime cur = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
        long mm = cur.toEpochSecond(ZoneOffset.ofHours(timeZone)) - start.toEpochSecond(ZoneOffset.ofHours(timeZone));
        String hexData = ByteUtils.completeBytes(Long.toHexString(mm), 8);
        this.responseContent.append(hexData);
        return this;
    }

    /**
     * 添加指定长度字节
     *
     * @param val
     * @return
     */
    private CommandResponse appendBytes(long val, int lenght) {
        String hexData = ByteUtils.completeBytes(Long.toHexString(val), lenght);
        this.responseContent.append(hexData);
        return this;
    }

    public PackageMark getPackageMark() {
        return packageMark;
    }

    public void setPackageMark(PackageMark packageMark) {
        this.packageMark = packageMark;
    }

    public String getNid() {
        return nid;
    }

    public void setNid(String nid) {
        this.nid = nid;
    }

    public int getContentLength() {
        return contentLength;
    }

    public void setContentLength(int contentLength) {
        this.contentLength = contentLength;
    }

    public String getEncoding() {
        return StringUtils.isNotBlank(this.encoding) ? encoding : Constant.DEFAULT_CHARSET;
    }

    public void setEncoding(String encoding) {
        this.encoding = encoding;
        this.charset = StringUtils.isNotBlank(encoding) ? Charset.forName(encoding) : Charset.forName(Constant.DEFAULT_CHARSET);
    }

    public StringBuffer getResponseContent() {
        return responseContent;
    }

    public void setResponseContent(StringBuffer responseContent) {
        this.responseContent = responseContent;
    }
}
