package com.jhscale.meter.wifi.model;

import com.jhscale.common.utils.DateUtils;
import com.jhscale.meter.utils.ByteUtils;
import com.jhscale.meter.utils.Constant;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.Date;

/**
 * @author lie_w
 * @title: ContentAssembler
 * @projectName jhscale-agreement
 * @description: Content 拼装
 * @date 2019/9/1523:50
 */
public class ContentAssembler {

    private StringBuffer content = new StringBuffer();
    private Charset charset;

    public ContentAssembler() {
    }

    public ContentAssembler(String charset) {
        if (StringUtils.isBlank(charset))
            charset = Constant.DEFAULT_CHARSET;
        this.charset = Charset.forName(charset);
    }

    /**
     * @Description 直接添加不转换
     * @Author wanglie
     * @Date 22:27 2019/3/27
     **/
    public ContentAssembler append(String part) {
        this.content.append(part);
        return this;
    }

    /**
     * @Description 直接添加不转换
     * @Author wanglie
     * @Date 22:27 2019/3/27
     **/
    public ContentAssembler append(char character) {
        this.content.append(ByteUtils.int2HexWithPush(Integer.valueOf(character), 2));
        return this;
    }

    /**
     * @Description 添加颠倒转换
     * @Author wanglie
     * @Date 22:27 2019/3/27
     **/
    public ContentAssembler appendConvert(String part) {
        this.content.append(ByteUtils.convert(part));
        return this;
    }

    /**
     * @Description 直接添加转换
     * @Author wanglie
     * @Date 22:27 2019/3/27
     **/
    public ContentAssembler appendHex(String data) {
        this.content.append(ByteUtils.int2Hex(Integer.parseInt(data)));
        return this;
    }

    /**
     * @Description 添加不定长文本
     * @Author wanglie
     * @Date 22:27 2019/3/27
     **/
    public ContentAssembler appendText(String part) {
        this.content.append(ByteUtils.toHexString(part.getBytes(charset))).append("00");
        return this;
    }

    /**
     * @Description 添加BigDecimal String
     * @Author wanglie
     * @Date 23:10 2019/3/27
     **/
    public ContentAssembler appendBigDecimal(String data) {
        String bigDecimal = String.valueOf(ByteUtils.toSPF(new BigDecimal(data)));
        return append4Bytes(bigDecimal);
    }

    /**
     * @Description 添加1个字节长度
     * @Author wanglie
     * @Date 13:45 2019/3/28
     **/
    public ContentAssembler append1Bytes(String data) {
        this.content.append(ByteUtils.int2HexWithPush(Integer.parseInt(data), 2));
        return this;
    }

    /**
     * @Description 转换成1字节 Integer
     * @Author wanglie
     * @Date 22:38 2019/3/27
     **/
    public ContentAssembler append1Bytes(int val) {
        this.content.append(ByteUtils.int2HexWithPush(val, 2));
        return this;
    }

    /**
     * @Description 转换成1字节 Integer
     * @Author wanglie
     * @Date 22:38 2019/3/27
     **/
    public ContentAssembler append4Integer(int val) {
        this.content.append(ByteUtils.int2HexWithPush(val, 8));
        return this;
    }


    /**
     * @Description 转换成2字节String
     * @Author wanglie
     * @Date 22:38 2019/3/27
     **/
    public ContentAssembler append2Bytes(String data) {
        this.content.append(ByteUtils.int2HexWithPush(Integer.parseInt(data), 4));
        return this;
    }

    /**
     * @Description 转换成2字节 Integer
     * @Author wanglie
     * @Date 22:38 2019/3/27
     **/
    public ContentAssembler append2Bytes(int val) {
        this.content.append(ByteUtils.int2HexWithPush(val, 4));
        return this;
    }

    /**
     * @Description 转换成4字节String
     * @Author wanglie
     * @Date 23:01 2019/3/27
     **/
    public ContentAssembler append4Bytes(String data) {
        this.content.append(ByteUtils.int2HexWithPush(Integer.parseInt(data), 8));
        return this;
    }

    /**
     * @Description 转换成4字节String
     * @Author wanglie
     * @Date 23:01 2019/3/27
     **/
    public ContentAssembler append(Date date) {
        long timeDefault = DateUtils.getDefaultTime().getTime();
        long timeLong = DateUtils.getLongTime(date, "");
        long seconds = (timeLong - timeDefault) / 1000;
        this.content.append(ByteUtils.completeBytes(Long.toHexString(seconds), 8));
        return this;
    }

    /**
     * @Description content拼装结束
     * @Author wanglie
     * @Date 3:43 2019/3/28
     **/
    public String over() {
        return this.content.toString();
    }
}
