package com.jhscale.meter.wifi.model;

import com.jhscale.common.utils.Base64Utils;
import com.jhscale.meter.exp.MeterException;
import com.jhscale.meter.exp.MeterStateEnum;
import com.jhscale.meter.utils.*;
import com.jhscale.meter.wifi.constant.CommandType;
import com.jhscale.meter.wifi.entity.*;
import org.apache.commons.lang3.StringUtils;
import org.spongycastle.util.encoders.Hex;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: WifiCmdUtils
 * @projectName jhscale-agreement
 * @description: Wifi 命令工具
 * @date 2020/6/2014:43
 */
public class WifiCmdUtils {

    /**
     * 组装
     *
     * @param handler
     * @param objects
     * @return
     */
    public static CMD assembler(String handler, JHObject... objects) {
        return assemblerByContent(handler, assembler(objects));
    }

    /**
     * @description: 拼装内容信息
     **/
    public static String assembler(JHObject... objects) {
        StringBuffer content = new StringBuffer();
        if (objects != null && objects.length > 0) {
            for (JHObject object : objects) {
                Content _content = (Content) object;
                if (!_content.canAdd()) continue;
                DataPart dataPart = new DataPart(_content);
                String mark = dataPart.getdCmd();
                if (dataPart.getdSort().length() / 2 == 1) {
                    mark += "0";
                } else {
                    mark += "1";
                }

                String dId = Integer.toBinaryString(dataPart.getdId().length() / 2);
                for (int i = 0; i < 3; i++) {
                    if (dId.length() % 3 != 0) {
                        dId = "0" + dId;
                    }
                }
                mark += dId;

                int len = dataPart.getdSort().length() + dataPart.getdId().length() + dataPart.getdData().length();
                String hexLen = Integer.toHexString(len / 2);
                if (hexLen.length() % 2 != 0) {
                    hexLen = "0" + hexLen;
                }
                String dLen = ByteUtils.convert(hexLen);

                String markLen = Integer.toBinaryString(dLen.length() / 2);
                if (markLen.length() % 2 != 0) {
                    markLen = "0" + markLen;
                }
                mark += markLen;

                String dMark = Integer.toString(Integer.parseInt(mark, 2), 16);
                content.append(dMark)
                        .append(dLen)
                        .append(dataPart.getdSort())
                        .append(dataPart.getdId())
                        .append(dataPart.getdData());
            }
        }
        return content.toString();
    }

    /**
     * @description: 内容拼装CMD
     **/
    public static CMD assemblerByContent(String handler, String content) {
        String nid = null;
        StringBuffer sb = new StringBuffer();

        sb.append(Constant.TOP);

        int dataLength = content.length() / 2;

        // 拼装Mark
        sb.append(new PackageMark(false, false, "00", CommandType.API, dataLength).bulidMark());

        // 获取nid
        nid = NidManagerUtils.getInstance().getNId(handler);
        sb.append(nid);

        // 添加长度
        String hexLength = Integer.toHexString(dataLength);
        if (hexLength.length() % 2 != 0) {
            hexLength = "0" + hexLength;
        }
        hexLength = ByteUtils.convert(hexLength);
        sb.append(hexLength);

        // 添加Content
        String contentData = ByteUtils.toHexAscii(ByteUtils.fromHexString(content.toString()));
        sb.append(contentData);
        sb.append(Constant.END);
        return new CMD(nid, sb);
    }

    /**
     * 应答组装
     *
     * @param response
     * @return
     */
    public static String response(CommandResponse response, String aeskey) throws MeterException {
        String hexLength = Integer.toHexString(response.getResponseContent().length() / 2);
        if (hexLength.length() % 2 != 0) hexLength = "0" + hexLength;
        hexLength = ByteUtils.toHexString(ByteUtils.reverse(ByteUtils.fromHexString(hexLength)));

        // 数据包内容
        String content = response.getResponseContent().toString();

        StringBuffer sb = new StringBuffer();
        sb.append(Constant.TOP);

        response.getPackageMark().setAck(true);
        response.getPackageMark().setLenBytes(response.getResponseContent().length());
        sb.append(response.getPackageMark().bulidMark())
                .append(response.getNid())
                .append(hexLength);

        if (response.getPackageMark().isEncrypt()) {
            if (StringUtils.isBlank(aeskey)) throw new MeterException(MeterStateEnum.AES_KEY_不存在);
            content = ByteUtils.appendFill(content);
            content = ByteUtils.toHexAscii(AesHexUtils.encrypt(Hex.decode(content), aeskey));
            sb.append(content);

            String crc = Integer.toHexString(CrcUtils.crc(ByteUtils.fromHexString(response.getResponseContent().toString())));

            int crcLength = crc.length();
            if (crcLength < 4) {
                for (int i = 0; i < (4 - crcLength); i++) {
                    crc = "0" + crc;
                }
            }

            sb.append(ByteUtils.convertHex(crc));
        } else {
            sb.append(content);
        }

        // 结尾
        sb.append(Constant.END);
        return sb.toString();
    }

    /**
     * 解析
     *
     * @param data    待解析数据
     * @param charter 字符集
     * @param base64  是否Base64
     * @return
     */
    @Deprecated
    public static List<Content> parse(String data, String charter, boolean base64) {
        return contentParse(data, charter, base64);
    }

    /**
     * @param bytes
     * @param charter
     * @description: 内容解析
     */
    public static List<Content> contentParse(byte[] bytes, String charter) {
        return contentParse(ByteUtils.toHexString(bytes), charter, false);
    }

    /**
     * 解析
     *
     * @param data    数据
     * @param charter 字符串
     * @param base64  是否BASE64压缩
     * @return
     */
    public static List<Content> contentParse(String data, String charter, boolean base64) {
        if (base64) data = Base64Utils.ungzipString(data);
        if (StringUtils.isBlank(charter)) charter = Constant.DEFAULT_CHARSET;
        List<DataPart> dataParts = new ArrayList<>();
        while (StringUtils.isNotBlank(data)) {
            ContentMark contentMark = new ContentMark(data.substring(0, 2));
            data = data.substring(2);
            String dLen = ByteUtils.convert(data.substring(0, contentMark.getLen()));
            int dataLen = Integer.parseInt(dLen, 16) * 2;
            String sort_Id_Data = data.substring(contentMark.getLen(), contentMark.getLen() + dataLen);
            data = data.substring(contentMark.getLen() + sort_Id_Data.length(), data.length());
            String dSort = sort_Id_Data.substring(0, contentMark.getSort());
            int dIdLen = Integer.parseInt(Integer.toString(contentMark.getId(), 10)) * 2;
            String dId = sort_Id_Data.substring(contentMark.getSort(), contentMark.getSort() + dIdLen);
            String dData = sort_Id_Data.substring(contentMark.getSort() + dIdLen, sort_Id_Data.length());
            dataParts.add(new DataPart(contentMark.getCmd(), dSort, dId, dData));
        }

        List<Content> contents = new ArrayList<>();
        if (dataParts != null && !dataParts.isEmpty()) {
            String finalCharter = charter;
            dataParts.forEach(dataPart -> {
                Content content = getContent(dataPart, finalCharter);
                if (Objects.nonNull(content)) contents.add(content);
            });
        }
        return contents;
    }

    /**
     * @param data
     * @param charter
     * @description: 解析 默认未压缩
     */
    public static List<Content> contentParse(String data, String charter) {
        return contentParse(data, charter, false);
    }

    /**
     * CMD AES解密
     *
     * @param bytes
     * @param aeskey
     * @return
     */
    public static CommandRequest cmdParse(byte[] bytes, String aeskey) throws MeterException {
        CommandRequest commandInfo = new CommandRequest(bytes);
        PackageMark packageMark = new PackageMark(bytes[1]);
        commandInfo.setPackageMark(packageMark);

        if (packageMark.getLenBytes() > 0) {
            commandInfo.setNid(ByteUtils.toHexAscii(Arrays.copyOfRange(bytes, 2, 4)));
            int contentLength = Integer.parseInt(ByteUtils.toHexAscii(ByteUtils.reverse(Arrays.copyOfRange(bytes, 4, 4 + packageMark.getLenBytes()))), 16);
            commandInfo.setContentLength(contentLength);

            if (packageMark.isEncrypt()) {
                if (StringUtils.isBlank(aeskey)) throw new MeterException(MeterStateEnum.AES_KEY_不存在);
                System.out.println(ByteUtils.toHexString(Arrays.copyOfRange(bytes, 4 + packageMark.getLenBytes(), bytes.length - 3)));
                byte[] datas = AesHexUtils.decrypt(Arrays.copyOfRange(bytes, 4 + packageMark.getLenBytes(), bytes.length - 3), aeskey);
                byte[] content = Arrays.copyOfRange(datas, 0, contentLength);
                System.out.println(ByteUtils.toHexString(content));
                commandInfo.setContent(content);
                commandInfo.setFilling(Arrays.copyOfRange(datas, contentLength, datas.length));
                commandInfo.setCrc(Integer.parseInt(ByteUtils.toHexAscii(ByteUtils.reverse(Arrays.copyOfRange(bytes, bytes.length - 3, bytes.length - 1))), 16));
                commandInfo.setCrcr(commandInfo.getCrc() == CrcUtils.crc(content));
            } else {
                commandInfo.setContent(Arrays.copyOfRange(bytes, 4 + packageMark.getLenBytes(), 4 + packageMark.getLenBytes() + contentLength));
            }
        }

        switch (packageMark.getCd()) {
            case CmdTypeConstant.CMD_CD:
                commandInfo.setType(CommandType.CMD);
                commandInfo.setService(commandInfo.lengthField(2));
                break;
            case CmdTypeConstant.DATA_CD:
                commandInfo.setType(CommandType.DATA);
                break;
            case CmdTypeConstant.API_CD:
                commandInfo.setType(CommandType.API);
                break;
            default:
                break;
        }
        return commandInfo;
    }

    /**
     * CMD AES解密
     *
     * @param cmd
     * @param aesKey
     * @return
     */
    public static CommandRequest cmdParse(String cmd, String aesKey) throws MeterException {
        return cmdParse(ByteUtils.fromHexString(cmd), aesKey);
    }

    /**
     * @description: 获取Content段
     **/
    private static Content getContent(DataPart dataPart, String charter) {
        Content content;
        switch (dataPart.getdSort().toUpperCase()) {
            case "01":
                content = new Log(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "02":
                content = new Derpartment(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "03":
                content = new Group(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "04":
                content = new PLU(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "05":
                content = new BarCode(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "06":
                content = new Put(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "07":
                content = new Assistant(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "08":
                content = new Data(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "09":
                content = new Text(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "0A":
                content = new Keyboard(dataPart.getdId(), dataPart.getdData(), charter, false);
                break;
            case "FF":
                content = new Derpartment(dataPart.getdId(), dataPart.getdData(), charter);
                break;
            default:
                content = null;
                break;
        }
        return content;
    }

    /**
     * @description: 商品列表
     **/
    public static List<Goods> getGoodsList(ContentParse parse, int number) {
        List<Goods> goodsList = new ArrayList<>();
        for (int i = 0; i < number; i++) {
            goodsList.add(new Goods(parse));
        }
        return goodsList;
    }
}
