package com.jhscale.sds.socket.listener;

import com.jhscale.sds.socket.config.SocketManager;
import com.jhscale.sds.socket.service.SocketHandlerService;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import lombok.extern.slf4j.Slf4j;

/**
 * @author Loveven
 * @title: SocketHandler
 * @projectName sds
 * @description: TODO
 * @date 2020/2/611:00
 */
@Slf4j
public class SocketHandler extends ChannelInboundHandlerAdapter {

    private SocketHandlerService socketHandlerService;

    public SocketHandler(SocketHandlerService socketHandlerService) {
        this.socketHandlerService = socketHandlerService;
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        String uniqueKey = ctx.channel().remoteAddress().toString();
        socketHandlerService.getSocketEventService().onReadListener(ctx, uniqueKey, msg);
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        if (SocketManager.getInstance().isAllowConnection()) {
            log.info("connection -> {}", ctx);
            SocketManager.getInstance().addClient(ctx.channel());
            String uniqueKey = ctx.channel().remoteAddress().toString();

            //将数据存储到distribute下
            try {
                socketHandlerService.create(uniqueKey);
                socketHandlerService.getSocketEventService().onConnectionListener(ctx, uniqueKey);
            } catch (Exception e) {
                log.error("error - connection - > " + e.getMessage());
                ctx.close();
            }

        } else {
            log.error("not allow connection -> {} ,nowConnectionSize -> {} ,maxConnectionSize -> {}",
                    ctx, SocketManager.getInstance().getNowConnection(), SocketManager.getInstance().getMaxConnection());
            ctx.close();
        }
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        log.debug("disconnection -> {}", ctx);
        String uniqueKey = ctx.channel().remoteAddress().toString();
        socketHandlerService.getSocketEventService().onDisConnectionListener(ctx, uniqueKey);

        SocketManager.getInstance().removeClient(ctx.channel());

        try {
            //将数据从distribute下异常
            socketHandlerService.remove(uniqueKey);
        } catch (Exception e) {
            log.error("error - disconnection -> {}", e.getMessage(), e);
        }
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        //心跳配置
        if (IdleStateEvent.class.isAssignableFrom(evt.getClass()) && socketHandlerService.getSocketEventService().hasOpenHeartCheck()) {
            IdleStateEvent event = (IdleStateEvent) evt;
            String uniqueKey = ctx.channel().remoteAddress().toString();
            if (event.state() == IdleState.READER_IDLE) {
                //表示已经多久没有收到数据了
                socketHandlerService.getSocketEventService().onHeartNoReadDataListener(ctx, uniqueKey);

            } else if (event.state() == IdleState.WRITER_IDLE) {
                //表示已经多久没有发送数据了
                socketHandlerService.getSocketEventService().onHeartNoWriteDataListener(ctx, uniqueKey);

            } else if (event.state() == IdleState.ALL_IDLE) {
                //表示已经多久既没有收到也没有发送数据了

            }
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        socketHandlerService.getSocketEventService().onException(ctx, cause);
    }
}
