package com.jhscale.security.zuul.application.filter;

import com.jhscale.common.utils.JSONUtils;
import com.jhscale.security.application.client.SecurityApplicationClient;
import com.jhscale.security.application.client.ao.ApiFlowLimitInfo;
import com.jhscale.security.application.client.ao.VerifyApiFlowLimitReq;
import com.jhscale.security.component.cache.base.LocalCache;
import com.jhscale.security.component.consensus.RpcConstants;
import com.jhscale.security.component.consensus.SecurityConstants;
import com.jhscale.security.component.consensus.message.ApplicationInfo;
import com.jhscale.security.component.consensus.utils.HttpUtils;
import com.jhscale.security.component.zuul.FilterUtils;
import com.jhscale.security.component.zuul.ZuulComponentConstants;
import com.jhscale.security.component.zuul.exp.SecurityZuulException;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cloud.netflix.zuul.filters.Route;
import org.springframework.cloud.netflix.zuul.filters.RouteLocator;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Objects;

/**
 * @author lie_w
 * @title: ApplicationIdentificationFilter
 * @projectName Rely-On-Security
 * @description: TODO
 * @date 2020-11-119:51
 */
@Slf4j
@Component
public class ApplicationIdentificationFilter extends ZuulFilter {

    @Autowired
    private RouteLocator routeLocator;

    @Autowired
    private SecurityApplicationClient applicationClient;

    @Autowired
    private AntPathMatcher antPathMatcher;

    @Autowired
    @Qualifier("app-api-flow-limit")
    private LocalCache localCache;

    /**
     * to classify a filter by type. Standard types in Zuul are "pre" for pre-routing filtering,
     * "route" for routing to an origin, "post" for post-routing filters, "error" for error handling.
     * We also support a "static" type for static responses see  StaticResponseFilter.
     * Any filterType made be created or added and run by calling FilterProcessor.runFilters(type)
     *
     * @return A String representing that type
     */
    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    /**
     * filterOrder() must also be defined for a filter. Filters may have the same  filterOrder if precedence is not
     * important for a filter. filterOrders do not need to be sequential.
     *
     * @return the int order of a filter
     */
    @Override
    public int filterOrder() {
        return ZuulComponentConstants.APPLICATION_IDENTIFICATION_FILTER_ORDER;
    }

    /**
     * a "true" return from this method means that the run() method should be invoked
     *
     * @return true if the run() method should be invoked. false will not invoke the run() method
     */
    @Override
    public boolean shouldFilter() {
        RequestContext ctx = RequestContext.getCurrentContext();
        // 被之前的Filter强行终止
        if (ctx.get(HttpUtils.HARD_BREAK_FLAG) != null) return false;
        // 系统应用
        return ctx.get(ZuulComponentConstants.SYS_ADMIN_FLAG) == null;
    }

    /**
     * if shouldFilter() is true, this method will be invoked. this method is the core method of a ZuulFilter
     *
     * @return Some arbitrary artifact may be returned. Current implementation ignores it.
     * @throws ZuulException if an error occurs during execution.
     */
    @Override
    public Object run() throws ZuulException {
        log.debug("开始识别应用......");
        RequestContext ctx = RequestContext.getCurrentContext();

        try {
            // 0. 应用路由识别
            Route route = routeLocator.getMatchingRoute(ctx.getRequest().getRequestURI());
            if (Objects.isNull(route)) {
                log.warn("不存在的应用");
                FilterUtils.notFound("不存在的应用");
                return null;
            }

            // 1. 系统应用
            if (ZuulComponentConstants.SYSTEM_APP_ID_LIST.contains(route.getId())) {
                log.warn("系统应用： {}", route.getId());
                return null;
            }

            // 2. 应用限流检测
            if (localCache.contains(route.getId())) {
                for (Object limit : localCache.get(route.getId(), List.class)) {
                    ApiFlowLimitInfo info = (ApiFlowLimitInfo) limit;
                    if ((!StringUtils.hasText(info.getHttpMethod()) || info.getHttpMethod().equals(ctx.getRequest().getMethod()))
                            && antPathMatcher.match(info.getApiPattern(), ctx.getRequest().getRequestURI())) {
                        try {
                            applicationClient.verifyApiFlowLimit(new VerifyApiFlowLimitReq(route.getId(), ctx.getRequest().getMethod(), ctx.getRequest().getRequestURI()));
                        } catch (Exception e) {
                            FilterUtils.forbidden(e.getMessage(), RpcConstants.APP_API_FLOW_LIMIT);
                            return null;
                        }
                        break;
                    }
                }
            }

            ApplicationInfo applicationInfo = new ApplicationInfo(route.getId());

            // 3. 设置应用信息到HTTP 请求头
            ctx.addZuulRequestHeader(SecurityConstants.APPLICATION_INFO_HEADER, JSONUtils.objectToUTF8Base64Code(applicationInfo));

            // 4. 设置应用识别成功标志
            ctx.set(ZuulComponentConstants.APPLICATION_IDENTIFICATION_FLAG, applicationInfo);
            ctx.set(ZuulComponentConstants.APPLICATION_ID, applicationInfo.getAppId());
            log.debug("应用识别结果：{}", applicationInfo);
        } catch (Exception e) {
            log.error("应用识别异常：{}", e.getMessage(), e);
            FilterUtils.fail(RpcConstants.APP_IDENTIFY_ERROR, ctx, new SecurityZuulException(e.getMessage()));
        }
        return null;
    }
}
