package com.jhscale.security.zuul.application.filter;

import com.jhscale.security.application.client.ao.SpecialApplicationInfo;
import com.jhscale.security.component.cache.base.LocalCache;
import com.jhscale.security.component.consensus.RpcConstants;
import com.jhscale.security.component.consensus.message.ApplicationInfo;
import com.jhscale.security.component.consensus.message.SsoUserInfo;
import com.jhscale.security.component.tools.utils.Jsons;
import com.jhscale.security.component.zuul.FilterUtils;
import com.jhscale.security.component.zuul.ZuulComponentConstants;
import com.jhscale.security.component.zuul.exp.SecurityZuulException;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Objects;

/**
 * @author lie_w
 * @title: SpecialApplicationFilter
 * @projectName Rely-On-Security
 * @description: TODO
 * @date 2020-11-119:58
 */
@Slf4j
@Component
public class SpecialApplicationFilter extends ZuulFilter {

    @Autowired
    @Qualifier("special-application-info")
    private LocalCache localCache;

    /**
     * to classify a filter by type. Standard types in Zuul are "pre" for pre-routing filtering,
     * "route" for routing to an origin, "post" for post-routing filters, "error" for error handling.
     * We also support a "static" type for static responses see  StaticResponseFilter.
     * Any filterType made be created or added and run by calling FilterProcessor.runFilters(type)
     *
     * @return A String representing that type
     */
    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    /**
     * filterOrder() must also be defined for a filter. Filters may have the same  filterOrder if precedence is not
     * important for a filter. filterOrders do not need to be sequential.
     *
     * @return the int order of a filter
     */
    @Override
    public int filterOrder() {
        return ZuulComponentConstants.BIZ_SERVER_ROUTE_FILTER_ORDER;
    }

    /**
     * a "true" return from this method means that the run() method should be invoked
     *
     * @return true if the run() method should be invoked. false will not invoke the run() method
     */
    @Override
    public boolean shouldFilter() {
        RequestContext ctx = RequestContext.getCurrentContext();
        if (ctx.get(ZuulComponentConstants.VERIFY_TOKEN_FLAG) == null) return false;
        return ctx.get(ZuulComponentConstants.APPLICATION_IDENTIFICATION_FLAG) != null;
    }

    /**
     * if shouldFilter() is true, this method will be invoked. this method is the core method of a ZuulFilter
     *
     * @return Some arbitrary artifact may be returned. Current implementation ignores it.
     * @throws ZuulException if an error occurs during execution.
     */
    @Override
    public Object run() throws ZuulException {
        RequestContext ctx = RequestContext.getCurrentContext();

        try {
            SsoUserInfo ssoUserInfo = (SsoUserInfo) ctx.get(ZuulComponentConstants.VERIFY_TOKEN_FLAG);
            ApplicationInfo applicationInfo = (ApplicationInfo) ctx.get(ZuulComponentConstants.APPLICATION_IDENTIFICATION_FLAG);

            // 1. 用户不存在等级信息，退出
            if (Objects.isNull(ssoUserInfo.getUserLevel())) {
                log.warn("用户: {} 不存在等级信息", Jsons.toJsonString(ssoUserInfo));
                return null;
            }

            // 2. 本地缓存不存在该应用信息，退出
            if (!localCache.contains(applicationInfo.getAppId())) {
                log.warn("本地缓存不存在该应用[{}]信息", Jsons.toJsonString(applicationInfo));
                return null;
            }

            // 3. 本地缓存不存在改专用通道，退出
            Map specialApplicationInfoMap = localCache.get(applicationInfo.getAppId(), Map.class);
            if (!specialApplicationInfoMap.containsKey(ssoUserInfo.getUserLevel())) {
                log.warn("本地缓存不存此专用通道");
                return null;
            }

            SpecialApplicationInfo specialApp = (SpecialApplicationInfo) specialApplicationInfoMap.get(ssoUserInfo.getUserLevel());
            // 4. 特殊的注册中心ID
            if (specialApp.getPathType() == RpcConstants.SEVER) {
                ctx.set(FilterConstants.SERVICE_ID_KEY, specialApp.getServerPath());
                log.warn("特殊的注册中心ID:{}", Jsons.toJsonString(specialApp));
                return null;
            }

            // 5. 特殊的具体服务地址
            ctx.set(ZuulComponentConstants.ROUTE_HOST, specialApp.getServerPath());
            log.debug("特殊的具体服务地址");
        } catch (Exception e) {
            log.error("用户特殊通道识别异常：{}", e.getMessage(), e);
            FilterUtils.fail(RpcConstants.FLOW_PATH_ERROR, ctx, new SecurityZuulException(e.getMessage()));
        }
        return null;
    }
}
