package com.jhscale.security.zuul.signature.filter;

import com.jhscale.common.exception.GeneralInternation;
import com.jhscale.common.exception.SignatureException;
import com.jhscale.common.model.http.JRequest;
import com.jhscale.common.utils.MD5Utils;
import com.jhscale.security.component.consensus.RpcConstants;
import com.jhscale.security.component.consensus.SecurityConstants;
import com.jhscale.security.component.consensus.exp.ConsenseException;
import com.jhscale.security.component.consensus.exp.ConsenseInternational;
import com.jhscale.security.component.consensus.utils.HeadUtils;
import com.jhscale.security.component.consensus.utils.HttpUtils;
import com.jhscale.security.component.zuul.FilterUtils;
import com.jhscale.security.component.zuul.ZuulComponentConstants;
import com.jhscale.security.zuul.signature.config.SignatureConfig;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import com.ysscale.framework.utils.JSONUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Objects;

/**
 * @author lie_w
 * @title: SignatureFilter
 * @projectName jhscale-security-start-parent
 * @description: TODO
 * @date 2020-12-0416:52
 */
@Slf4j
@Component
public class SignatureFilter extends ZuulFilter {

    @Autowired
    private SignatureConfig signatureConfig;

    /**
     * to classify a filter by type. Standard types in Zuul are "pre" for pre-routing filtering,
     * "route" for routing to an origin, "post" for post-routing filters, "error" for error handling.
     * We also support a "static" type for static responses see  StaticResponseFilter.
     * Any filterType made be created or added and run by calling FilterProcessor.runFilters(type)
     *
     * @return A String representing that type
     */
    @Override
    public String filterType() {
        return FilterConstants.PRE_TYPE;
    }

    /**
     * filterOrder() must also be defined for a filter. Filters may have the same  filterOrder if precedence is not
     * important for a filter. filterOrders do not need to be sequential.
     *
     * @return the int order of a filter
     */
    @Override
    public int filterOrder() {
        return ZuulComponentConstants.SIGNATURE_ORDER_FILTER_ORDER;
    }

    /**
     * a "true" return from this method means that the run() method should be invoked
     *
     * @return true if the run() method should be invoked. false will not invoke the run() method
     */
    @Override
    public boolean shouldFilter() {
        RequestContext ctx = RequestContext.getCurrentContext();
        if (Objects.nonNull(ctx.get(HttpUtils.HARD_BREAK_FLAG))) return false;
        return signatureConfig.isOpen() && Objects.nonNull(ctx.get(SecurityConstants.EQUIPMENT_KEY));
    }

    /**
     * if shouldFilter() is true, this method will be invoked. this method is the core method of a ZuulFilter
     *
     * @return Some arbitrary artifact may be returned. Current implementation ignores it.
     * @throws ZuulException if an error occurs during execution.
     */
    @Override
    public Object run() throws ZuulException {
        RequestContext ctx = RequestContext.getCurrentContext();
        HttpServletRequest request = ctx.getRequest();
        String key = RequestContext.getCurrentContext().get(SecurityConstants.EQUIPMENT_KEY).toString();
        log.debug("开始签名验证 Key:{}", key);

        String method = request.getMethod();
        try {
            if ("GET".equalsIgnoreCase(method)) {
                String secret = HeadUtils.getHeadContent(request, SecurityConstants.EQUIPMENT_SECRET);
                if (StringUtils.isBlank(secret))
                    throw new ConsenseException(RpcConstants.EQUIPMENT_EXCEPTION, ConsenseInternational.设备密钥不存在);
                if (!secret.equals(key))
                    throw new ConsenseException(RpcConstants.EQUIPMENT_EXCEPTION, ConsenseInternational.设备密钥无效);
            } else {
                String data = HttpUtils.readData(request);
                log.debug("请求方式：{}，参数内容：{}", method, data);

                if (StringUtils.isBlank(data))
                    throw new ConsenseException(RpcConstants.EQUIPMENT_EXCEPTION, ConsenseInternational.请求参数不存在);

                JRequest jsonRequest = JSONUtils.jsonToPojo(data, JRequest.class);
                if (Objects.isNull(jsonRequest))
                    throw new ConsenseException(RpcConstants.EQUIPMENT_EXCEPTION, ConsenseInternational.请求参数无效);

                jsonRequest.checkPackage(key, signatureConfig.getSingnatureInvalidTime());

                String sourceSignature = MD5Utils.bulidSourceSignature(JSONUtils.jsonToPojo(data, Map.class), key);
                log.debug("服务器解析签名源码：{}", sourceSignature);
                String sign = MD5Utils.bulidMD5Signature(sourceSignature);
                log.debug("服务器还原MD5签名：{}，请求签名：{}", sign, jsonRequest.getSign());
                if (!jsonRequest.getSign().equals(sign))
                    throw new SignatureException(GeneralInternation.签名无效);
            }
        } catch (ConsenseException e) {
            log.error("签名参数异常：{}", e.getMessage());
            FilterUtils.fail(e.getResponseCode(), ctx, e);
        } catch (SignatureException e) {
            log.error("签名检查异常：{}", e.getMessage());
            FilterUtils.fail(RpcConstants.EQUIPMENT_EXCEPTION, ctx, e);
        }
        return null;
    }
}
