package com.jhscale.wxpay.client;

import com.jhscale.wxpay.config.WxPayConfig;
import com.jhscale.wxpay.model.WxPayReq;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContexts;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.MediaType;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.SSLContext;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.List;

/**
 * @author lie_w
 * @title: WxPayClient
 * @projectName rely-on-pay
 * @description: TODO
 * @date 2019/11/312:56
 */
@Slf4j
public class WxPayClient {

    private RestTemplate restTemplate;

    private WxPayConfig wxPayConfig;

    public WxPayClient(WxPayConfig wxPayConfig) throws Exception {
        this.wxPayConfig = wxPayConfig;
        buildRestTemplate();
    }

    /**
     * 发起请求
     *
     * @param request 请求对象
     * @param tClass  响应对象的类型
     * @param <T>     响应对象的类型
     * @return 响应对象
     */
    public <T> T execute(WxPayReq request, Class<T> tClass) {
        request.initConfig(wxPayConfig);
        request.randomNonceStr();
        request.initSign(wxPayConfig.getKey());
        String nonceStr = request.getNonce_str();
        log.debug("id:{},post-url =>{}", nonceStr, request.getUrl(wxPayConfig.getChannel()));
        T result = (T) request.sendReq(restTemplate, wxPayConfig, tClass);
        log.debug("id:{},post-res=>{}", nonceStr, result);
        return result;
    }

    /**
     * 发起请求
     *
     * @param request 请求对象
     * @param <T>     响应对象的类型
     * @return 响应对象
     */
    public <T> T execute(WxPayReq request) {
        return (T) execute(request, request.getResClass());
    }

    private class WxMappingJackson2HttpMessageConverter extends MappingJackson2XmlHttpMessageConverter {
        private WxMappingJackson2HttpMessageConverter() {
            List<MediaType> mediaTypes = new ArrayList<>();
            mediaTypes.add(MediaType.TEXT_PLAIN);
            mediaTypes.add(MediaType.MULTIPART_FORM_DATA);
            mediaTypes.add(MediaType.APPLICATION_JSON);
            setSupportedMediaTypes(mediaTypes);
        }
    }

    private void buildRestTemplate() throws Exception {
        if (wxPayConfig == null) {
            throw new RuntimeException("wxPayConfig was empty.");
        }

        //构建https协议
        if (StringUtils.isNotEmpty(wxPayConfig.getCertLocalPath())) {
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            //加载本地的证书进行https加密传输
//            FileInputStream fileInputStream = new FileInputStream(new File(wxPayConfig.getCertLocalPath()));
            InputStream fileInputStream = new ClassPathResource(wxPayConfig.getCertLocalPath()).getInputStream();//加载本地的证书进行https加密传输
            //设置证书密码
            keyStore.load(fileInputStream, wxPayConfig.getCertPassword().toCharArray());
            SSLContext sslContext = SSLContexts.custom().loadKeyMaterial(keyStore, wxPayConfig.getCertPassword().toCharArray()).build();

            SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE);

            CloseableHttpClient httpClient = HttpClients.custom()
                    .setSSLSocketFactory(sslConnectionSocketFactory)
                    .build();
            ClientHttpRequestFactory httpClientRequestFactory = new HttpComponentsClientHttpRequestFactory(httpClient);
            restTemplate = new RestTemplate(httpClientRequestFactory);
        } else {
            restTemplate = new RestTemplate();
        }
        restTemplate.getMessageConverters().add(0, new StringHttpMessageConverter(Charset.forName("UTF-8")));
        restTemplate.getMessageConverters().add(new WxPayClient.WxMappingJackson2HttpMessageConverter());
    }
}
