package com.jhscale.wxpay.req;

import com.jhscale.wxpay.model.WxPayReq;
import com.jhscale.wxpay.res.ApplymentMicroSubmitRes;
import com.jhscale.wxpay.utils.RSAEncryptUtil;
import lombok.Data;

/**
 * 小微申请入驻
 * https://pay.weixin.qq.com/wiki/doc/api/xiaowei.php?chapter=19_2
 */
@Data
public class ApplymentMicroSubmitReq implements WxPayReq<ApplymentMicroSubmitRes> {
    @Override
    public String getUrl(String channel) {
        return channel + "/applyment/micro/submit";
    }

    /**
     * 接口版本号 [是] 固定版本号为3.0
     */
    private String version = "3.0";

    /**
     * 平台证书序列号 [是] 用于对敏感信息进行加密的平台证书序列号，获取方法详见平台证书及序列号获取接口（接口返回参数"serial_no"即为平台证书序列号）
     */
    private String cert_sn;

    /**
     * 商户号 [是] 服务商的商户号
     */
    private String mch_id;

    /**
     * 随机字符串 [是] 自定义随机字符串，可参照随机数生成算法
     */
    private String nonce_str;

    /**
     * 签名类型 [是] 签名类型，仅支持HMAC-SHA256 。该字段需参与签名sign的计算。
     */
    private String sign_type = "HMAC-SHA256";

    /**
     * 签名 [是] 通过签名计算后得到的签名值，详见签名生成算法
     */
    private String sign;

    /**
     * 业务申请编号 [是] 服务商自定义的商户唯一编号。每个编号对应一个申请单，每个申请单审核通过后会生成一个微信支付商户号。
     */
    private String business_code;

    /**
     * 身份证人像面照片 [是] 请填写由图片上传接口预先上传图片生成好的media_id
     */
    private String id_card_copy;

    /**
     * 身份证国徽面照片 [是] 请填写由图片上传接口预先上传图片生成好的media_id
     */
    private String id_card_national;

    /**
     * 身份证姓名 [是] 请填写小微商户本人身份证上的姓名，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    public String id_card_name;

    /**
     * 身份证号码 [是] 15位数字 或  17位数字+1位数字|X ，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String id_card_number;

    /**
     * 身份证有效期限 [是] ["1970-01-01","长期"]
     * 1.注意参照示例中的格式
     * 2.结束时间需大于开始时间
     * 3.要与上传的身份证照片内容一致
     */
    private String id_card_valid_time;

    /**
     * 开户名称 [是] 必须与身份证姓名一致，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String account_name;

    /**
     * 开户银行 [是] 详细参见开户银行对照表
     */
    private String account_bank;

    /**
     * 开户银行省市编码 [是] 至少精确到市,详细参见微信支付提供省市对照表
     */
    private String bank_address_code;

    /**
     * 开户银行全称（含支行）[否]
     * 1）17家直连银行无需填写，其他银行请务必填写
     * 2）需填写银行全称，如"深圳农村商业银行XXX支行"
     * 详细参见开户银行全称（含支行）对照表
     */
    private String bank_name;

    /**
     * 银行账号 [是] 数字，长度遵循系统支持的对私卡号长度要求,该字段需进行加密处理，加密方法详见敏感信息加密说明。小微商户开户目前不支持以下前缀的银行卡
     */
    private String account_number;

    /**
     * 门店名称 最长50个中文字符 [是]
     * 门店场所：填写门店名称
     * 流动经营/便民服务：填写经营/服务名称
     * 线上商品/服务交易：填写线上店铺名称
     */
    private String store_name;

    /**
     * 门店省市编码 详细参见微信支付提供省市对照表 [是]
     * 门店场所：填写门店省市编码
     * 流动经营/便民服务：填写经营/服务所在地省市编码
     * 线上商品/服务交易：填写卖家所在地省市编码
     */
    private String store_address_code;

    /**
     * 门店街道名称 最长500个中文字符（无需填写省市信息） [是]
     * 门店场所：填写店铺详细地址，具体区/县及街道门牌号或大厦楼层
     * 流动经营/便民服务：填写“无"
     * 线上商品/服务交易：填写电商平台名称
     */
    private String store_street;

    /**
     * 门店经度 [否] 数字或小数
     */
    private String store_longitude;

    /**
     * 门店纬度 [否] 数字或小数
     */
    private String store_latitude;

    /**
     * 门店门口照片 [是] 请填写已预先上传图片生成好的MediaID
     * 门店场所：提交门店门口照片，要求招牌清晰可见
     * 流动经营/便民服务：提交经营/服务现场照片
     * 线上商品/服务交易：提交店铺首页截图
     */
    private String store_entrance_pic;

    /**
     * 店内环境照片 [是] 请填写已预先上传图片生成好的MediaID
     * 门店场所：提交店内环境照片
     * 流动经营/便民服务：可提交另一张经营/服务现场照片
     * 线上商品/服务交易：提交店铺管理后台截图
     */
    private String indoor_pic;

    /**
     * 经营场地证明 [否] 请填写已预先上传图片生成好的MediaID，门面租赁合同扫描件或经营场地证明（需与身份证同名）
     */
    private String address_certification;

    /**
     * 商户简称 [是] UTF-8格式，中文占3个字节，即最多16个汉字长度。将在支付完成页向买家展示，需与商家的实际经营场景相符
     */
    private String merchant_shortname;

    /**
     * 客服电话 [是] UTF-8格式，中文占3个字节，即最多16个汉字长度。在交易记录中向买家展示，请确保电话畅通以便平台回拨确认
     */
    private String service_phone;

    /**
     * 售卖商品/提供服务描述 请填写以下描述之一： [是]
     * 餐饮
     * 线下零售
     * 居民生活服务
     * 休闲娱乐
     * 交通出行
     * 其他
     */
    private String product_desc;

    /**
     * 费率 [是] 由服务商指定，微信支付提供字典值
     */
    private String rate;

    /**
     * 补充说明 [否] 可填写需要额外说明的文字
     */
    private String business_addition_desc;

    /**
     * 补充材料 [否] 最多可上传5张照片，请填写已预先上传图片生成好的MediaID
     */
    private String business_addition_pics;

    /**
     * 超级管理员姓名 [是] 和身份证姓名一致 ，该字段需进行加密处理，加密方法详见敏感信息加密说明，超级管理员需在开户后进行签约，并可接收日常重要管理信息和进行资金操作，请确定其为商户法定代表人或负责人
     */
    private String contact;

    /**
     * 手机号码 [是] 11位数字，手机号码 ，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String contact_phone;

    /**
     * 联系邮箱 [否] 需要带@，遵循邮箱格式校验 ，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String contact_email;

    /**
     * 隐私加密证书 [是] 隐私加密使用
     */
    private String cert;

    @Override
    public Class<ApplymentMicroSubmitRes> getResClass() {
        return ApplymentMicroSubmitRes.class;
    }

    @Override
    public String[] ignoreFileds() {
        return new String[]{"cert"};
    }

    public void setId_card_name(String id_card_name) {
        this.id_card_name = RSAEncryptUtil.rsaEncrypt(id_card_name, cert);
    }

    public void setId_card_number(String id_card_number) {
        this.id_card_number = RSAEncryptUtil.rsaEncrypt(id_card_number, cert);
    }

    public void setAccount_name(String account_name) {
        this.account_name = RSAEncryptUtil.rsaEncrypt(account_name, cert);
    }

    public void setContact(String contact) {
        this.contact = RSAEncryptUtil.rsaEncrypt(contact, cert);
    }

    public void setContact_phone(String contact_phone) {
        this.contact_phone = RSAEncryptUtil.rsaEncrypt(contact_phone, cert);
    }

    public void setContact_email(String contact_email) {
        this.contact_email = RSAEncryptUtil.rsaEncrypt(contact_email, cert);
    }
}
