package com.jhscale.wxpay.req;

import com.jhscale.wxpay.model.WxPayReq;
import com.jhscale.wxpay.res.ApplymentMicroSubmitupgradeRes;
import com.jhscale.wxpay.utils.RSAEncryptUtil;
import lombok.Data;

/**
 * 小微提交升级申请单接口
 * https://pay.weixin.qq.com/wiki/doc/api/mch_xiaowei.php?chapter=28_2&index=2
 */
@Data
public class ApplymentMicroSubmitupgradeReq implements WxPayReq<ApplymentMicroSubmitupgradeRes> {
    @Override
    public String getUrl(String channel) {
        return channel + "/applyment/micro/submitupgrade";
    }

    /**
     * 接口版本号 [是] 默认值1.0
     */
    private String version = "1.0";

    /**
     * 商户号 [是] 服务商的商户号
     */
    private String mch_id;

    /**
     * 随机字符串 [是] 参考签名规则
     */
    private String nonce_str;

    /**
     * 签名类型 [是] 签名类型，仅支持HMAC-SHA256
     */
    private String sign_type = "HMAC-SHA256";

    /**
     * 签名 [是] 参考签名规则
     */
    private String sign;

    /**
     * 平台证书序列号 [是] 用于对敏感信息进行加密的平台证书序列号，获取方法详见平台证书及序列号获取接口（接口返回参数"serial_no"即为平台证书序列号）
     */
    private String cert_sn;

    /**
     * 小微商户号 [是] 小微商户的商户号（小微申请状态需为待签约或完成）
     */
    private String sub_mch_id;

    /**
     * 主体类型 [是]
     * 2-企业   4-个体工商户
     * 3-党政、机关及事业单位  1708-其他组织
     */
    private String organization_type;

    /**
     * 营业执照扫描件 [是] 请填写通过图片上传接口预先上传图片生成好的MediaID
     * 仅能上传1张图片
     */
    private String business_license_copy;

    /**
     * 营业执照注册号 [是] 请填写营业执照上的营业执照注册号
     * 若主体类型为个体工商户或企业，注册号格式须为15位数字或 18位数字|大写字母
     */
    private String business_license_number;

    /**
     * 商户名称 [是] 请填写营业执照上的商户名称
     * 2~110个字符，支持括号
     * 个体工商户不能以“公司”结尾
     */
    private String merchant_name;

    /**
     * 注册地址 [是] 256个字以内
     */
    private String company_address;

    /**
     * 经营者姓名/法定代表人 [是] 请填写营业执照上的经营者/法人姓名
     * 必须与小微商户联系人相同
     * 该字段需进行加密处理，加密方法详见敏感信息加密说明。
     */
    private String legal_person;

    /**
     * 营业期限 [是] 请填写营业执照上的营业期限
     * 注意参照示例中的格式
     * 结束时间需大于开始时间
     * 有效期必须大于60天
     * ["1970-01-01","长期"]
     */
    private String business_time;

    /**
     * 营业执照类型 [是] 1762-已三证合一    1763-未三证合一
     * 若营业执照上的营业执照注册号为18位统一社会信用代码，请选择“已三证合一”，否则请选择“未三证合一”
     * 注意：参数为business_licence_type，这里为licence，非license
     */
    private String business_licence_type;

    /**
     * 组织机构代码证照片 [否] 请填写通过图片上传接口预先上传图片生成好的MediaID
     * 仅能上传1张图片
     * 主体类型为企业/党政、机关及事业单位/其他组织 且 营业执照为未三证合一，必填
     */
    private String organization_copy;

    /**
     * 组织机构代码 [否] 请填写组织机构代码证上的组织机构代码
     * 9或10位 数字|字母|连字符
     * 主体类型为企业/党政、机关及事业单位/其他组织 且 营业执照为未三证合一，必填
     */
    private String organization_number;

    /**
     * 组织机构代码有效期限 [否] 请填写组织机构代码证上的组织机构代码有效期
     * 注意参照示例中的格式
     * 结束时间需大于开始时间
     * 有效期必须大于60天
     * 主体类型为企业/党政、机关及事业单位/其他组织 且 营业执照为未三证合一，必填
     * ["1970-01-01","长期"]
     */
    private String organization_time;

    /**
     * 开户名称 [否] 请填写对公账户，开户名称必须与营业执照上的“商户名称”一致
     * 主体类型为企业/党政、机关及事业单位/其他组织时，必填
     * 该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String account_name;

    /**
     * 开户银行 [否] 详细参见开户银行对照表
     * 主体类型为企业/党政、机关及事业单位/其他组织时，必填
     */
    private String account_bank;

    /**
     * 开户银行省市编码 [否] 至少精确到市,详细参见省市区编号对照表
     * 主体类型为企业/党政、机关及事业单位/其他组织时，必填
     */
    private String bank_address_code;

    /**
     * 开户银行全称（含支行） [否]
     * 17家直连银行无需填写，其他银行请务必填写
     * 需填写银行全称，如"深圳农村商业银行XXX支行"
     * 详细参见开户银行全称（含支行）对照表
     * 主体类型为企业/党政、机关及事业单位/其他组织 且 开户银行为非直连银行时，必填
     */
    private String bank_name;

    /**
     * 银行卡号 [否] 数字，长度遵循系统支持的对公卡号长度要求
     * 主体类型为企业/党政、机关及事业单位/其他组织时，必填
     * 该字段需进行加密处理，加密方法详见敏感信息加密说明。
     */
    private String account_number;

    /**
     * 商户简称 [是] 2~30个中文字符、英文字符、符号。将在支付完成页向买家展示，需与微信经营范围相关
     */
    private String merchant_shortname;

    /**
     * 费率结算规则ID [是] 详细参见费率结算规则对照表
     */
    private String business;

    /**
     * 特殊资质 [否]
     * 1）视行业而定，详细参见费率结算规则对照表
     * 2）注意示例中的格式
     * 3）最多可上传5张照片，请填写通过图片上传接口预先上传图片生成好的MediaID
     * ["xxx","xxx"]
     */
    private String qualifications;

    /**
     * 经营场景 [是]
     * 1721-线下  1837-公众号  1838-小程序  1724-APP  1840-PC网站
     * 可多选，至少选择1个
     * [1721,1724]
     */
    private String business_scene;

    /**
     * 补充说明 [否]
     */
    private String business_addition_desc;

    /**
     * 补充材料 [否] 最多可上传5张照片，请填写通过图片上传接口预先上传图片生成好的MediaID
     * 请使用JSON格式的数组
     * ["123","456"]
     */
    private String business_addition_pics;

    /**
     * 联系邮箱 [否] 联系邮箱
     * 需要带@，遵循邮箱格式校验 ，若小微商户入驻时没有提交邮箱，该字段必填，该字段需进行加密处理，加密方法详见敏感信息加密说明
     */
    private String contact_email;

    /**
     * 1、默认授权服务商可帮商家发起-Native支付、JSAPI支付、付款码支付。
     * 2、当字段business_scene填写包括：1837-公众号、1838-小程序、1724-APP、1840-PC，请按以下规则填写资料；部分场景审批通过后可额外获得对应的支付权限。
     */

    /**
     * 公众号APPID [否]
     * 1、若经营场景包括公众号场景，则必填；
     * 2、可输入“已绑定服务商商户号的公众号APPID”或“与商家主体一致且已认证的公众号APPID”，二选一填写；公众号APPID需是已认证的服务号、政府或媒体类型的订阅号；
     * 3、若填写商家公众号APPID，则审核通过后，可由商家发起- JSAPI支付。
     * 注：商家入驻后，需自行完成商户号与公众号APPID的关联，才可正常发起交易
     */
    private String mp_appid;

    /**
     * 公众号页面截图 [否]
     * 若经营场景包含公众号场景，可填写最多五张图片,非必填
     * 请提供展示商品/服务的页面截图/设计稿（最多5张）
     * 请填写已预先上传图片生成好的MediaID
     * 请使用JSON格式的数组
     * ["xxx","xxx"]
     */
    private String mp_app_screen_shots;

    /**
     * 小程序APPID [否]
     * 1、若经营场景包括小程序场景，必填
     * 2、可输入“已绑定服务商商户号的小程序APPID”或“与商家主体一致且已认证的小程序APPID”，二选一填写
     * 3、若填写商家小程序APPID，则审核通过后，可由商家发起-JSAPI支付。
     * 注：商家入驻后，需自行完成商户号与小程序APPID的关联，才可正常发起交易
     */
    private String miniprogram_appid;

    /**
     * 小程序页面截图 [否]
     * 若经营场景包含小程序场景，非必填
     * 请提供展示商品/服务的页面截图/设计稿（最多5张）
     * 请填写已预先上传图片生成好的MediaID
     * 请使用JSON格式的数组
     * ["xxx","xxx"]
     */
    private String miniprogram_screen_shots;

    /**
     * 应用APPID [否]
     * 1、若经营场景包括APP场景，必填
     * 2、可输入“已绑定服务商商户号的应用APPID”或“与商家主体一致且已认证的应用APPID”，二选一填写
     * 3、若填写服务商应用APPID，审核通过后，可由服务商帮商家发起-APP支付
     * 4、若填写商家应用APPID，审核通过后，可由服务商帮商家发起或商家自行发起-APP支付。
     * 注：商家入驻后，需自行完成商户号与应用APPID的关联，才可正常发起交易
     */
    private String app_appid;

    /**
     * APP截图 [否]
     * 若经营场景包含APP场景，必填
     * 请提供展示商品/服务的页面截图/设计稿（最多5张）
     * 请填写已预先上传图片生成好的MediaID
     * 请使用JSON格式的数组
     * ["xxx","xxx"]
     */
    private String app_screen_shots;

    /**
     * APP下载链接 [否]
     * 若经营场景包含APP场景，非必填
     * 若app已上线，请提供主流应用市场（如app store，应用宝，豌豆荚）下载链接，提供安装包无效
     */
    private String app_download_url;

    /**
     * PC网站域名 [否]
     * 若经营场景包含PC网站场景，必填
     * 网站域名需ICP备案
     */
    private String web_url;

    /**
     * 网站授权函 [否]
     * 若经营场景包含PC网站场景，且网站域名的备案主体与申请主体不同，必填
     * 网站授权函需加盖公章
     */
    private String web_authoriation_letter;

    /**
     * PC网站对应的公众号APPID [否]
     * 1、若经营场景包括PC网站场景，选填
     * 2、可填写“与商家主体一致且已认证的公众号、小程序、应用的APPID”，其中公众号APPID需是已认证的服务号、政府或媒体类型的订阅号
     * 3、审核通过后，可由商家发起-JSAPI支付、Native支付。注：商家入驻后，需自行完成商户号与对应APPID的关联，才可正常发起交易
     */
    private String web_appid;

    /**
     * 隐私加密证书 [是] 隐私加密使用
     */
    private String cert;

    @Override
    public Class<ApplymentMicroSubmitupgradeRes> getResClass() {
        return ApplymentMicroSubmitupgradeRes.class;
    }

    @Override
    public String[] ignoreFileds() {
        return new String[]{"cert"};
    }

    public void setLegal_person(String legal_person) {
        this.legal_person = RSAEncryptUtil.rsaEncrypt(legal_person, cert);
    }

    public void setAccount_name(String account_name) {
        this.account_name = RSAEncryptUtil.rsaEncrypt(account_name, cert);
    }

    public void setAccount_number(String account_number) {
        this.account_number = RSAEncryptUtil.rsaEncrypt(account_number, cert);
    }

    public void setContact_email(String contact_email) {
        this.contact_email = RSAEncryptUtil.rsaEncrypt(contact_email, cert);
    }
}
