package com.jhscale.wxpay.req;

import com.jhscale.wxpay.config.WxPayConfig;
import com.jhscale.wxpay.model.WxPayReq;
import com.jhscale.wxpay.res.UploadmediaRes;
import com.jhscale.wxpay.utils.XMLUtils;
import com.ysscale.framework.utils.FileUtil;
import lombok.Data;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HTTP;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.util.EntityUtils;
import org.springframework.core.io.ClassPathResource;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.SSLContext;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.*;
import java.security.cert.CertificateException;

/**
 * 图片上传API
 * 接口地址: https://pay.weixin.qq.com/wiki/doc/api/mch_xiaowei.php?chapter=19_9
 */
@Data
public class UploadmediaReq implements WxPayReq<UploadmediaRes> {
    @Override
    public String getUrl(String channel) {
        return channel + "/secapi/mch/uploadmedia";
    }

    /**
     * 商户号 [是] 服务商商户号或渠道号
     */
    private String mch_id;

    /**
     * 媒体文件 [是] form-data中媒体文件标识，有filename、filelength、content-type等信息。不参与签名计算
     */
    private String media;

    /**
     * 媒体文件内容hash值 [是] 根据媒体文件内容进行MD5计算后的值，注意小写 调用 md5HashCode 方法计算
     */
    private String media_hash;

    /**
     * 签名 [是] 通过签名算法计算得出的签名值，详见签名生成算法
     */
    private String sign;

    /**
     * 签名方式 [否] HMAC-SHA256加密方式，其他或者不填为MD5方式
     */
    private String sign_type = "HMAC-SHA256";

    /**
     * 随机字符串 [否] 配合结构使用
     */
    private String nonce_str;

    @Override
    public Class<UploadmediaRes> getResClass() {
        return UploadmediaRes.class;
    }

    /**
     * 对上次文件进行MD5获取其Hash值
     *
     * @param inputStream
     * @return
     */
    public static String md5HashCode(InputStream inputStream) {
        try {
            MessageDigest MD5 = MessageDigest.getInstance("MD5");
            byte[] buffer = new byte[8192];
            int length;
            while ((length = inputStream.read(buffer)) != -1) {
                MD5.update(buffer, 0, length);
            }
            return new String(Hex.encodeHex(MD5.digest()));
        } catch (NoSuchAlgorithmException | IOException e) {
            e.printStackTrace();
            return "";
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public String[] ignoreFileds() {
        return new String[]{"media", "url", "body"};
    }

    @Override
    public void randomNonceStr() {
    }

    /**
     * 文件下载地址 [是] 全路径地址
     */
    private String url;

    private byte[] body;

    @Override
    public void beforeInit() {
        if (body != null && body.length > 0) return;
        body = FileUtil.getFileByte(url);
        this.media_hash = md5HashCode(new ByteArrayInputStream(body));
    }

    @Override
    public Object sendReq(RestTemplate restTemplate, WxPayConfig wxPayConfig, Class tClass) {
        String res = null;
        try {
            MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
            multipartEntityBuilder.addTextBody("mch_id", mch_id, ContentType.MULTIPART_FORM_DATA);
            multipartEntityBuilder.addBinaryBody("media", body, ContentType.WILDCARD, this.getMedia());
            multipartEntityBuilder.addTextBody("media_hash", media_hash, ContentType.MULTIPART_FORM_DATA);
            multipartEntityBuilder.addTextBody("sign_type", sign_type, ContentType.MULTIPART_FORM_DATA);
            multipartEntityBuilder.addTextBody("sign", sign, ContentType.MULTIPART_FORM_DATA);

            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            ClassPathResource resource = new ClassPathResource(wxPayConfig.getCertLocalPath());
            InputStream inputStream = resource.getInputStream();
            keyStore.load(inputStream, this.getMch_id().toCharArray());// 这里写密码..默认是你的MCHID
            SSLContext sslcontext = SSLContexts.custom().loadKeyMaterial(keyStore, wxPayConfig.getMch_id().toCharArray()).build();
            SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslcontext, new String[]{"TLSv1"}, null, SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER);
            CloseableHttpClient httpclient = HttpClients.custom().setSSLSocketFactory(sslsf).build();
            HttpPost httpPost = new HttpPost(getUrl(wxPayConfig.getChannel()));
            RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(10000).setConnectionRequestTimeout(10000).setSocketTimeout(10000).build();
            httpPost.setConfig(requestConfig);
            ////这里的Content-type要设置为"multipart/form-data"，否则返回“参数填写有误，请检查后重试”
            httpPost.addHeader(HTTP.CONTENT_TYPE, "multipart/form-data; charset=UTF-8");
            httpPost.addHeader(HTTP.USER_AGENT, "wxpay sdk java v1.0 " + wxPayConfig);
            httpPost.setEntity(multipartEntityBuilder.build());
            CloseableHttpResponse response = httpclient.execute(httpPost);
            res = EntityUtils.toString(response.getEntity(), "UTF-8");
        } catch (KeyStoreException | IOException | NoSuchAlgorithmException | CertificateException | UnrecoverableKeyException | KeyManagementException e) {
            UploadmediaRes uploadmediaRes = new UploadmediaRes();
            uploadmediaRes.setReturn_code("FAIL");
            uploadmediaRes.setReturn_msg("微信文件上传失败");
            return uploadmediaRes;
        }

        if (StringUtils.isNoneBlank(res) && XMLUtils.isXmlDocument(res)) {
            return XMLUtils.convertXmlStrToObject(UploadmediaRes.class, res);
        }
        return null;
    }
}
