package com.jhscale.pay.service.impl;

import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.domain.*;
import com.alipay.api.request.*;
import com.alipay.api.response.*;
import com.jhscale.pay.config.PayConfig;
import com.jhscale.pay.req.*;
import com.jhscale.pay.res.*;
import com.jhscale.pay.service.AuthService;
import com.jhscale.pay.service.OrderPayService;
import com.jhscale.pay.util.PayMessageCode;
import com.ysscale.framework.orderem.DeviceHandleStateEnum;
import com.ysscale.framework.utils.DateUtils;
import com.ysscale.framework.utils.JSONUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

/**
 * @author lie_w
 * @title: AliPayServiceImpl
 * @projectName ysscale-core
 * @description: TODO
 * @date 2019/11/2321:57
 */
@Service("1")
@SuppressWarnings("all")
public class AliPayServiceImpl implements AuthService, OrderPayService {

    private static final Logger LOGGER = LoggerFactory.getLogger(AliPayServiceImpl.class);

    private static final String PAY_SGIN = "ALI:";

    @Autowired
    private PayConfig payConfig;

    @Autowired
    private AlipayClient client;

    /**
     * @param payAuthorizationReq
     * @description: 支付宝令牌换取
     */
    @Override
    public AliPayAuthorizationRes payAuthorization(AliPayAuthorizationReq req) {
        AlipayOpenAuthTokenAppRequest request = new AlipayOpenAuthTokenAppRequest();

        AlipayOpenAuthTokenAppModel model = new AlipayOpenAuthTokenAppModel();
        model.setGrantType(req.getAuthType());
        model.setCode(req.getAuthCode());
        request.setBizModel(model);

        LOGGER.debug("pay:支付宝支付授权>>>>>>" + "授权码:" + req.getAuthCode() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayOpenAuthTokenAppResponse response = null;
        try {
            response = client.execute(request);
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("pay:支付宝支付授权>>>>>>" + "授权码:" + req.getAuthCode() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        AliPayAuthorizationRes res = new AliPayAuthorizationRes(PAY_SGIN);
        if ("10000".equals(response.getCode()) && "Success".equals(response.getMsg())) {
            String app_auth_token = response.getAppAuthToken();
            String app_refresh_token = response.getAppRefreshToken();
            String user_id = response.getUserId();
            long expires_in = Long.parseLong(response.getExpiresIn());  //令牌有效期
            int days = (int) expires_in / 86400;
            Date date = DateUtils.addDate(new Date(), days);
            res.setpId(user_id);
            res.setPayAliToken(app_auth_token);
            res.setRefreshAliToken(app_refresh_token);
            res.setAliTokenCreate(date);
            res.setHandleState(DeviceHandleStateEnum.SUCCESS);
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TOKEN_INVALID);
            res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                    response.getSubMsg() : response.getMsg());
        }
        return res;
    }

    /**
     * @param createOrderReq
     * @description: 创建订单
     */
    @Override
    public CreateOrderRes createOrder(CreateOrderReq req) {
        LOGGER.debug("cloudpay : 支付宝创建订单>>>>>> {}", JSONUtils.objectJsonParse(req));
        AlipayTradePayRequest request = new AlipayTradePayRequest();
        AlipayTradePayModel model = new AlipayTradePayModel();
        model.setOutTradeNo(req.getOrderNo());
        model.setScene("bar_code");
        model.setAuthCode(req.getPayCode());
        model.setSubject(req.getBody());
        model.setTotalAmount(req.getTotalFee().toString());
        model.setOperatorId(req.getAdminId());
        model.setStoreId(req.getStoreId());
        model.setTerminalId(req.getMac());
        model.setTimeoutExpress(payConfig.getAliDefaultCloseTime());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝创建订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + JSONUtils.objectJsonParse(request));
        AlipayTradePayResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝创建订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        CreateOrderRes res = new CreateOrderRes(PAY_SGIN);
        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setTradeNo(response.getOutTradeNo());
                res.setCashFee(new BigDecimal(response.getTotalAmount()));
                res.setActualFee(new BigDecimal(response.getBuyerPayAmount()));
            } else if ("10003".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.WAIT);
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_ERROR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

    /**
     * @param queryOrderReq
     * @description: 查询订单
     */
    @Override
    public QueryOrderRes queryOrder(QueryOrderReq req) {
        LOGGER.debug("cloudpay:支付宝查询订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayTradeQueryRequest request = new AlipayTradeQueryRequest();

        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(req.getOrderNo());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝查询订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + JSONUtils.objectJsonParse(request));
        AlipayTradeQueryResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝查询订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        QueryOrderRes res = new QueryOrderRes(PAY_SGIN);

        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                switch (response.getTradeStatus()) {
                    case "WAIT_BUYER_PAY":
                        res.setHandleState(DeviceHandleStateEnum.WAIT);
                        break;
                    case "TRADE_CLOSED":
                        res.setHandleState(DeviceHandleStateEnum.FAIL);
                        res.setErrCode(PayMessageCode.TRADE_ERROR);
                        res.setErrMsg("未付款交易超时关闭，或支付完成后全额退款");
                        break;
                    case "TRADE_SUCCESS":
                        res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                        res.setTradeNo(response.getOutTradeNo());
                        res.setCashFee(new BigDecimal(response.getTotalAmount()));
                        res.setActualFee(new BigDecimal(response.getBuyerPayAmount()));
                        break;
                    case "TRADE_FINISHED":
                        res.setHandleState(DeviceHandleStateEnum.FAIL);
                        res.setErrCode(PayMessageCode.TRADE_ERROR);
                        res.setErrMsg("交易结束，不可退款");
                        break;
                    default:
                        break;
                }
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_ERROR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

    /**
     * @param req
     * @description: 订单退款
     */
    @Override
    public RefundOrderRes refundOrder(RefundOrderReq req) {
        LOGGER.debug("cloudpay:支付宝订单退款>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayTradeRefundRequest request = new AlipayTradeRefundRequest();

        AlipayTradeRefundModel model = new AlipayTradeRefundModel();
        model.setOutTradeNo(req.getOrderNo());
        model.setRefundAmount(req.getRefundFee().toString());
        model.setOutRequestNo(req.getRefundNo());
        model.setRefundReason(req.getRefundReason());
        model.setOperatorId(req.getAdminId());
        model.setStoreId(req.getStoreId());
        model.setTerminalId(req.getMac());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝退款订单>>>>>>" + "订单编号:" + req.getOrderNo() + "，退款金额:" + req.getRefundFee() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayTradeRefundResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝退款订单>>>>>>" + "订单编号:" + req.getOrderNo() + "，退款金额:" + req.getRefundFee() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        RefundOrderRes res = new RefundOrderRes(PAY_SGIN);
        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setRefundTradeNo(response.getOutTradeNo());
                res.setRefundFee(new BigDecimal(response.getRefundFee()));
            } else if ("10003".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.WAIT);
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_REFUND_ERR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

    /**
     * @param req
     * @description: 查询订单退款
     */
    @Override
    public QueryRefundOrderRes queryRefundOrder(QueryRefundOrderReq req) {
        LOGGER.debug("cloudpay:支付宝查询订单退款>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayTradeFastpayRefundQueryRequest request = new AlipayTradeFastpayRefundQueryRequest();

        AlipayTradeFastpayRefundQueryModel model = new AlipayTradeFastpayRefundQueryModel();
        model.setOutTradeNo(req.getOrderNo());
        model.setOutRequestNo(req.getRefundNo());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝查询订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "，退款编号:" + req.getRefundNo() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayTradeFastpayRefundQueryResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝查询订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "，退款编号:" + req.getRefundNo() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        QueryRefundOrderRes res = new QueryRefundOrderRes();
        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setRefundTradeNo(response.getOutTradeNo());
                res.setRefundFee(new BigDecimal(response.getRefundAmount()));
            } else if ("10003".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.WAIT);
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_REFUND_ERR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return null;
    }

    /**
     * @param closeOrderReq
     * @description: 关闭订单
     */
    @Override
    public CloseOrderRes closeOrder(CloseOrderReq req) {
        LOGGER.debug("cloudpay:支付宝关闭订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayTradeCloseRequest request = new AlipayTradeCloseRequest();

        AlipayTradeCloseModel model = new AlipayTradeCloseModel();
        model.setOutTradeNo(req.getOrderNo());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝关闭订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayTradeCloseResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝关闭订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        CloseOrderRes res = new CloseOrderRes(PAY_SGIN);
        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_CLOSE_ERR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

    /**
     * @param req
     * @description: 撤销订单
     */
    @Override
    public CancelOrderRes cancelOrder(CancelOrderReq req) {
        LOGGER.debug("cloudpay:支付宝撤销订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayTradeCancelRequest request = new AlipayTradeCancelRequest();

        AlipayTradeCancelModel model = new AlipayTradeCancelModel();
        model.setOutTradeNo(req.getOrderNo());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝撤销订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayTradeCancelResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝撤销订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        CancelOrderRes res = new CancelOrderRes(PAY_SGIN);

        if (Objects.isNull(response) || "ACQ.SYSTEM_ERROR".equals(response.getSubCode())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setRetryFlag(response.getRetryFlag());
                res.setAction(response.getAction());
                res.setGmtRefundPay(response.getGmtRefundPay());
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_CANCEL_ERR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

    /**
     * @param req
     * @description: 账单下载
     */
    @Override
    public BillDownloadRes cancelOrder(BillDownloadReq req) {
        LOGGER.debug("cloudpay:支付宝账单下载>>>>>> {}", JSONUtils.objectJsonParse(req));

        AlipayDataDataserviceBillDownloadurlQueryRequest request = new AlipayDataDataserviceBillDownloadurlQueryRequest();

        AlipayDataDataserviceBillDownloadurlQueryModel model = new AlipayDataDataserviceBillDownloadurlQueryModel();
        model.setBillType(StringUtils.isBlank(req.getType()) ? "trade" : req.getType());
        model.setBillDate(req.getDate());
        request.setBizModel(model);

        LOGGER.debug("cloudpay:支付宝账单下载>>>>>>" + "类型:" + req.getType() + "/日期：" + req.getDate() + "/ 请求:" + JSONUtils.objectJsonParse(model));
        AlipayDataDataserviceBillDownloadurlQueryResponse response = null;
        try {
            response = client.execute(request, null, req.getToken());
        } catch (AlipayApiException e) {
            e.printStackTrace();
        }
        LOGGER.debug("cloudpay:支付宝账单下载>>>>>>" + "类型:" + req.getType() + "/日期：" + req.getDate() + "/ 响应:" + JSONUtils.objectJsonParse(response));

        BillDownloadRes res = new BillDownloadRes(PAY_SGIN);

        if (Objects.isNull(response)) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("10000".equals(response.getCode())) {
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setData(response.getBillDownloadUrl());
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_BILL_ERR);
                res.setErrMsg(StringUtils.isNotBlank(response.getSubMsg()) ?
                        response.getSubMsg() : response.getMsg());
            }
        }
        return res;
    }

}
