package com.jhscale.pay.service.impl;

import com.jhscale.pay.config.PayConfig;
import com.jhscale.pay.req.*;
import com.jhscale.pay.res.*;
import com.jhscale.pay.service.OrderPayService;
import com.jhscale.pay.util.PayMessageCode;
import com.jhscale.wxpay.client.WxPayClient;
import com.jhscale.wxpay.req.*;
import com.jhscale.wxpay.res.*;
import com.ysscale.framework.orderem.DeviceHandleStateEnum;
import com.ysscale.framework.utils.BigDecimalUtils;
import com.ysscale.framework.utils.DateUtils;
import com.ysscale.framework.utils.JSONUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Objects;

/**
 * @author lie_w
 * @title: WxPayServiceImpl
 * @projectName ysscale-core
 * @description: TODO
 * @date 2019/11/2321:58
 */
@Service("2")
@SuppressWarnings("all")
public class WxPayServiceImpl implements OrderPayService {

    private static final Logger LOGGER = LoggerFactory.getLogger(WxPayServiceImpl.class);

    private static final String PAY_SGIN = "WX:";

    @Autowired
    private PayConfig payConfig;

    @Autowired
    private WxPayClient client;

    /**
     * @param createOrderReq
     * @description: 创建订单
     */
    @Override
    public CreateOrderRes createOrder(CreateOrderReq req) {
        LOGGER.debug("cloudpay : 微信创建订单>>>>>> {}", JSONUtils.objectJsonParse(req));
        MicropayReq micropayReq = new MicropayReq();
        micropayReq.setSub_mch_id(req.getToken());
        micropayReq.setDevice_info(req.getMac());
        micropayReq.setBody(req.getBody());
        micropayReq.setOut_trade_no(req.getOrderNo());
        micropayReq.setTotal_fee(BigDecimalUtils.getDivideMoney(req.getTotalFee()).toString());
        micropayReq.setSpbill_create_ip(req.getIp());
        micropayReq.setAuth_code(req.getPayCode());
        micropayReq.setTime_start(DateUtils.getWxOrderStartTime(req.getCreateTime()));
        micropayReq.setTime_expire(DateUtils.getWxOrderCloseTime(req.getCreateTime(), payConfig.getWxDefaultCloseTime()));
        MicropayReq.SceneInfo sceneInfo = new MicropayReq.SceneInfo();
        sceneInfo.setId(req.getStoreId());
        sceneInfo.setName(req.getStoreName());
        sceneInfo.setArea_code(req.getAreaCode());
        micropayReq.setScene_info(JSONUtils.objectToJson(sceneInfo));

        LOGGER.debug("cloudpay : 微信创建订单>>>>>>" + " 请求:" + (JSONUtils.objectJsonParse(micropayReq)));
        MicropayRes execute = client.execute(micropayReq);
        LOGGER.debug("cloudpay : 微信创建订单>>>>>>" + " 响应:" + (JSONUtils.objectJsonParse(execute)));

        CreateOrderRes res = new CreateOrderRes(PAY_SGIN);
        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else {
            if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
                BigDecimal yuanMoney = BigDecimalUtils.getYuanMoney(new BigDecimal(execute.getCash_fee()));
                res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                res.setTradeNo(execute.getTransaction_id());
                res.setCashFee(yuanMoney);
                res.setActualFee(yuanMoney);
            } else if ("USERPAYING".equals(execute.getErr_code()) || "BANKERROR".equals(execute.getErr_code())) {
                res.setHandleState(DeviceHandleStateEnum.WAIT);
            } else {
                res.setHandleState(DeviceHandleStateEnum.FAIL);
                res.setErrCode(PayMessageCode.TRADE_ERROR);
                res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                        execute.getErr_code_des() : execute.getReturn_msg());
            }
        }
        return res;
    }

    /**
     * @param queryOrderReq
     * @description: 查询订单
     */
    @Override
    public QueryOrderRes queryOrder(QueryOrderReq req) {
        LOGGER.debug("cloudpay:微信查询订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        OrderQueryReq orderQueryReq = new OrderQueryReq();
        orderQueryReq.setSub_mch_id(req.getToken());
        orderQueryReq.setOut_trade_no(req.getOrderNo());

        LOGGER.debug("cloudpay:微信查询订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + (JSONUtils.objectJsonParse(orderQueryReq)));
        OrderQueryRes execute = client.execute(orderQueryReq);
        LOGGER.debug("cloudpay:微信查询订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + (JSONUtils.objectJsonParse(execute)));

        QueryOrderRes res = new QueryOrderRes(PAY_SGIN);
        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
            switch (execute.getTrade_state()) {
                case "SUCCESS":
                    BigDecimal yuanMoney = BigDecimalUtils.getYuanMoney(new BigDecimal(execute.getCash_fee()));
                    res.setHandleState(DeviceHandleStateEnum.SUCCESS);
                    res.setTradeNo(execute.getTransaction_id());
                    res.setCashFee(yuanMoney);
                    res.setActualFee(yuanMoney);
                    break;
                case "USERPAYING":
                    res.setHandleState(DeviceHandleStateEnum.WAIT);
                    break;
                default:
                    res.setHandleState(DeviceHandleStateEnum.FAIL);
                    res.setErrCode(PayMessageCode.TRADE_ERROR);
                    res.setErrMsg(StringUtils.isNotBlank(execute.getTrade_state_desc()) ?
                            execute.getTrade_state_desc() : execute.getErr_code_des());
                    break;
            }
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_ERROR);
            res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                    execute.getErr_code_des() : execute.getReturn_msg());
        }
        return res;
    }

    /**
     * @param req
     * @description: 订单退款
     */
    @Override
    public RefundOrderRes refundOrder(RefundOrderReq req) {
        LOGGER.debug("cloudpay:微信订单退款>>>>>> {}", JSONUtils.objectJsonParse(req));

        RefundReq refundReq = new RefundReq();
        refundReq.setOut_trade_no(req.getOrderNo());
        refundReq.setOut_refund_no(req.getRefundNo());
        refundReq.setTotal_fee(BigDecimalUtils.getDivideMoney(req.getCashFee()).toString());
        refundReq.setRefund_fee(BigDecimalUtils.getDivideMoney(req.getRefundFee()).toString());
        refundReq.setRefund_desc(req.getRefundReason());

        LOGGER.debug("cloudpay:微信订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + (JSONUtils.objectJsonParse(refundReq)));
        RefundRes execute = client.execute(refundReq);
        LOGGER.debug("cloudpay:微信订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + (JSONUtils.objectJsonParse(execute)));

        RefundOrderRes res = new RefundOrderRes(PAY_SGIN);
        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.SUCCESS);
            res.setRefundTradeNo(execute.getRefund_id());
            res.setRefundFee(BigDecimalUtils.getYuanMoney(new BigDecimal(execute.getSettlement_refund_fee())));
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_REFUND_ERR);
            res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                    execute.getErr_code_des() : execute.getReturn_msg());
        }
        return res;
    }

    /**
     * @param req
     * @description: 查询订单退款
     */
    @Override
    public QueryRefundOrderRes queryRefundOrder(QueryRefundOrderReq req) {
        LOGGER.debug("cloudpay:微信查询订单退款>>>>>> {}", JSONUtils.objectJsonParse(req));

        RefundQueryReq refundQueryReq = new RefundQueryReq();
        refundQueryReq.setOut_trade_no(req.getOrderNo());
        refundQueryReq.setOut_refund_no(req.getRefundNo());

        LOGGER.debug("cloudpay:微信查询订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + (JSONUtils.objectJsonParse(refundQueryReq)));
        RefundQueryRes execute = client.execute(refundQueryReq);
        LOGGER.debug("cloudpay:微信查询订单退款>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + (JSONUtils.objectJsonParse(execute)));

        QueryRefundOrderRes res = new QueryRefundOrderRes();
        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
            switch (execute.getRefund_status_$n()) {
                case "SUCCESS":
                    res.setRefundTradeNo(execute.getRefund_id_$n());
                    res.setRefundFee(BigDecimalUtils.getYuanMoney(new BigDecimal(execute.getSettlement_refund_fee_$n())));
                    break;
                case "PROCESSING":
                    res.setHandleState(DeviceHandleStateEnum.WAIT);
                    break;
                default:
                    res.setHandleState(DeviceHandleStateEnum.FAIL);
                    res.setErrCode(PayMessageCode.TRADE_REFUND_ERR);
                    res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                            execute.getErr_code_des() : execute.getReturn_msg());
                    break;
            }
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_REFUND_ERR);
            res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                    execute.getErr_code_des() : execute.getReturn_msg());
        }
        return res;
    }

    /**
     * @param closeOrderReq
     * @description: 关闭订单
     */
    @Override
    public CloseOrderRes closeOrder(CloseOrderReq req) {
        LOGGER.debug("cloudpay:微信关闭订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        OrderCloseReq orderCloseReq = new OrderCloseReq();
        orderCloseReq.setSub_mch_id(req.getToken());
        orderCloseReq.setOut_trade_no(req.getOrderNo());

        LOGGER.debug("cloudpay:微信关闭订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 请求:" + (JSONUtils.objectJsonParse(orderCloseReq)));
        OrderCloseRes execute = client.execute(orderCloseReq);
        LOGGER.debug("cloudpay:微信关闭订单>>>>>>" + "订单编号:" + req.getOrderNo() + "/ 响应:" + (JSONUtils.objectJsonParse(execute)));

        CloseOrderRes res = new CloseOrderRes(PAY_SGIN);
        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.SUCCESS);
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_CLOSE_ERR);
            res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                    execute.getErr_code_des() : execute.getReturn_msg());
        }
        return res;
    }

    /**
     * @param req
     * @description: 撤销订单
     */
    @Override
    public CancelOrderRes cancelOrder(CancelOrderReq req) {
        LOGGER.debug("cloudpay:微信撤销订单>>>>>> {}", JSONUtils.objectJsonParse(req));

        ReverseReq reverseReq = new ReverseReq();
        reverseReq.setSub_mch_id(req.getOrderNo());
        reverseReq.setOut_trade_no(req.getOrderNo());

        LOGGER.debug("cloudpay:微信撤销订单>>>>>>" + "子商户号:" + req.getToken() + ",商户订单号:" + req.getOrderNo() + "/ 请求:" + JSONUtils.objectJsonParse(reverseReq));
        ReverseRes execute = client.execute(reverseReq);
        LOGGER.debug("cloudpay:微信撤销订单>>>>>>" + "子商户号:" + req.getToken() + ",商户订单号:" + req.getOrderNo() + "/ 响应:" + JSONUtils.objectJsonParse(execute));

        CancelOrderRes res = new CancelOrderRes(PAY_SGIN);

        if (Objects.isNull(execute) || "SYSTEMERROR".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code()) && "SUCCESS".equals(execute.getResult_code())) {
            res.setHandleState(DeviceHandleStateEnum.SUCCESS);
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_CANCEL_ERR);
            res.setErrMsg(StringUtils.isNotBlank(execute.getErr_code_des()) ?
                    execute.getErr_code_des() : execute.getReturn_msg());
        }
        return res;
    }

    /**
     * @param req
     * @description: 账单下载
     */
    @Override
    public BillDownloadRes cancelOrder(BillDownloadReq req) {
        LOGGER.debug("cloudpay:微信账单下载>>>>>> {}", JSONUtils.objectJsonParse(req));

        DownloadBillReq downloadBillReq = new DownloadBillReq();
        downloadBillReq.setSub_mch_id(req.getToken());
        downloadBillReq.setBill_type(StringUtils.isBlank(req.getType()) ? "ALL" : req.getType());
        downloadBillReq.setBill_date(req.getDate());

        LOGGER.debug("cloudpay:微信账单下载>>>>>>" + "类型:" + req.getType() + "/日期：" + req.getDate() + "/ 请求:" + JSONUtils.objectToJson(downloadBillReq));
        DownloadBillRes execute = client.execute(downloadBillReq);
        LOGGER.debug("cloudpay:微信账单下载>>>>>>" + "类型:" + req.getType() + "/日期：" + req.getDate() + "/ 响应:" + JSONUtils.objectToJson(execute));

        BillDownloadRes res = new BillDownloadRes(PAY_SGIN);

        if (Objects.isNull(execute)) {
            res.setHandleState(DeviceHandleStateEnum.WAIT);
        } else if ("SUCCESS".equals(execute.getReturn_code())) {
            res.setHandleState(DeviceHandleStateEnum.SUCCESS);
            res.setData(execute.getData());
        } else {
            res.setHandleState(DeviceHandleStateEnum.FAIL);
            res.setErrCode(PayMessageCode.TRADE_BILL_ERR);
            res.setErrMsg(execute.getReturn_msg());
        }
        return res;
    }
}
