/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.governator.lifecycle;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.netflix.governator.lifecycle.AnnotationFinder;
import com.netflix.governator.lifecycle.ClasspathUrlDecoder;
import com.netflix.governator.lifecycle.DirectoryClassFilter;
import java.io.File;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClasspathScanner {
    private static final Logger log = LoggerFactory.getLogger(ClasspathScanner.class);
    protected final ClassLoader classLoader;
    private final Set<Class<?>> classes;
    private final Set<Constructor> constructors;
    private final Set<Method> methods;
    private final Set<Field> fields;

    public ClasspathScanner(Collection<String> basePackages, Collection<Class<? extends Annotation>> annotations) {
        this(basePackages, annotations, Thread.currentThread().getContextClassLoader());
    }

    public ClasspathScanner(Collection<String> basePackages, Collection<Class<? extends Annotation>> annotations, ClassLoader classLoader) {
        Preconditions.checkNotNull(annotations, (Object)"annotations cannot be null");
        Preconditions.checkNotNull((Object)classLoader, (Object)"classLoader cannot be null");
        log.debug("Starting classpath scanning...");
        this.classLoader = classLoader;
        HashSet localClasses = Sets.newHashSet();
        HashSet localConstructors = Sets.newHashSet();
        HashSet localMethods = Sets.newHashSet();
        HashSet localFields = Sets.newHashSet();
        this.doScanning(basePackages, annotations, localClasses, localConstructors, localMethods, localFields);
        this.classes = ImmutableSet.copyOf((Collection)localClasses);
        this.constructors = ImmutableSet.copyOf((Collection)localConstructors);
        this.methods = ImmutableSet.copyOf((Collection)localMethods);
        this.fields = ImmutableSet.copyOf((Collection)localFields);
        log.debug("Classpath scanning done");
    }

    public Set<Class<?>> getClasses() {
        return this.classes;
    }

    public Set<Constructor> getConstructors() {
        return this.constructors;
    }

    public Set<Method> getMethods() {
        return this.methods;
    }

    public Set<Field> getFields() {
        return this.fields;
    }

    protected void doScanning(Collection<String> basePackages, Collection<Class<? extends Annotation>> annotations, Set<Class<?>> localClasses, Set<Constructor> localConstructors, Set<Method> localMethods, Set<Field> localFields) {
        if (basePackages.isEmpty()) {
            log.warn("No base packages specified - no classpath scanning will be done");
            return;
        }
        log.info("Scanning packages : " + basePackages + " for annotations " + annotations);
        for (String basePackage : basePackages) {
            try {
                String basePackageWithSlashes = basePackage.replace(".", "/");
                Enumeration<URL> resources = this.classLoader.getResources(basePackageWithSlashes);
                while (resources.hasMoreElements()) {
                    URL url = resources.nextElement();
                    try {
                        if (this.isJarURL(url)) {
                            String jarPath = url.getFile();
                            if (jarPath.contains("!")) {
                                jarPath = jarPath.substring(0, jarPath.indexOf("!"));
                                url = new URL(jarPath);
                            }
                            File file = ClasspathUrlDecoder.toFile(url);
                            try {
                                JarFile jar = new JarFile(file);
                                Throwable throwable = null;
                                try {
                                    Enumeration<JarEntry> list = jar.entries();
                                    while (list.hasMoreElements()) {
                                        JarEntry entry = list.nextElement();
                                        try {
                                            if (!entry.getName().endsWith(".class") || !entry.getName().startsWith(basePackageWithSlashes)) continue;
                                            AnnotationFinder finder = new AnnotationFinder(this.classLoader, annotations);
                                            new ClassReader(jar.getInputStream(entry)).accept((ClassVisitor)finder, 1);
                                            this.applyFinderResults(localClasses, localConstructors, localMethods, localFields, finder);
                                        }
                                        catch (Exception e) {
                                            log.debug("Unable to scan JarEntry '{}' in '{}'. {}", new Object[]{entry.getName(), file.getCanonicalPath(), e.getMessage()});
                                        }
                                    }
                                }
                                catch (Throwable throwable2) {
                                    throwable = throwable2;
                                    throw throwable2;
                                }
                                finally {
                                    if (jar == null) continue;
                                    if (throwable != null) {
                                        try {
                                            jar.close();
                                        }
                                        catch (Throwable x2) {
                                            throwable.addSuppressed(x2);
                                        }
                                        continue;
                                    }
                                    jar.close();
                                }
                            }
                            catch (Exception e) {
                                log.debug("Unable to scan '{}'. {}", new Object[]{file.getCanonicalPath(), e.getMessage()});
                            }
                            continue;
                        }
                        DirectoryClassFilter filter = new DirectoryClassFilter(this.classLoader);
                        for (String className : filter.filesInPackage(url, basePackage)) {
                            AnnotationFinder finder = new AnnotationFinder(this.classLoader, annotations);
                            new ClassReader(filter.bytecodeOf(className)).accept((ClassVisitor)finder, 1);
                            this.applyFinderResults(localClasses, localConstructors, localMethods, localFields, finder);
                        }
                    }
                    catch (Exception e) {
                        log.debug("Unable to scan jar '{}'. {} ", new Object[]{url, e.getMessage()});
                    }
                }
            }
            catch (Exception e) {
                throw new RuntimeException("Classpath scanning failed for package '" + basePackage + "'", e);
            }
        }
    }

    private void applyFinderResults(Set<Class<?>> localClasses, Set<Constructor> localConstructors, Set<Method> localMethods, Set<Field> localFields, AnnotationFinder finder) {
        for (Class<?> cls : finder.getAnnotatedClasses()) {
            if (localClasses.contains(cls)) {
                log.debug(String.format("Duplicate class found for '%s'", cls.getCanonicalName()));
                continue;
            }
            localClasses.add(cls);
        }
        for (Method method : finder.getAnnotatedMethods()) {
            if (localMethods.contains(method)) {
                log.debug(String.format("Duplicate method found for '%s:%s'", method.getClass().getCanonicalName(), method.getName()));
                continue;
            }
            localMethods.add(method);
        }
        for (Constructor ctor : finder.getAnnotatedConstructors()) {
            if (localConstructors.contains(ctor)) {
                log.debug(String.format("Duplicate constructor found for '%s:%s'", ctor.getClass().getCanonicalName(), ctor.toString()));
                continue;
            }
            localConstructors.add(ctor);
        }
        for (Field field : finder.getAnnotatedFields()) {
            if (localFields.contains(field)) {
                log.debug(String.format("Duplicate field found for '%s:%s'", field.getClass().getCanonicalName(), field.toString()));
                continue;
            }
            localFields.add(field);
        }
    }

    private boolean isJarURL(URL url) {
        String protocol = url.getProtocol();
        return "zip".equals(protocol) || "jar".equals(protocol) || "file".equals(protocol) && url.getPath().endsWith(".jar");
    }
}

