package eleme.openapi.sdk.api.service;

import eleme.openapi.sdk.api.annotation.Service;
import eleme.openapi.sdk.api.base.BaseNopService;
import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.oauth.response.Token;
import eleme.openapi.sdk.config.Config;
import eleme.openapi.sdk.api.entity.cpc.*;
import eleme.openapi.sdk.api.enumeration.cpc.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.*;

/**
 * CPC竞价服务
 */
@Service("eleme.cpc")
public class CpcService extends BaseNopService {
    public CpcService(Config config,Token token) {
        super(config, token, CpcService.class);
    }

    /**
     * 查询余额
     *
     * @param shopId 店铺ID
     * @return 账户信息
     * @throws ServiceException 服务异常
     */
    public OAllBalance getAllBalance(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.getAllBalance", params);
    }

    /**
     * 确认店铺两证是否齐全
     *
     * @param shopId 店铺ID
     * @return 两证查询结果
     * @throws ServiceException 服务异常
     */
    public OShopCertificationResponse checkShopCertification(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.checkShopCertification", params);
    }

    /**
     * 查询推广信息
     *
     * @param shopId 店铺ID
     * @return 推广信息
     * @throws ServiceException 服务异常
     */
    public OWagerInformation getWagerInformation(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.getWagerInformation", params);
    }

    /**
     * 根据出价查询预估信息
     *
     * @param shopId 店铺ID
     * @param bid CPC出价
     * @return 预估信息
     * @throws ServiceException 服务异常
     */
    public OShopWagerEstimate getWagerEstimate(long shopId, double bid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("bid", bid);
        return call("eleme.cpc.getWagerEstimate", params);
    }

    /**
     * 查询推荐价格、预估信息
     *
     * @param shopId 店铺ID
     * @return CPC推荐价格、预估点击提升、预估日预算信息
     * @throws ServiceException 服务异常
     */
    public OSuggestWagerInfo getSuggestWagerInfo(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.getSuggestWagerInfo", params);
    }

    /**
     * 查询推广修改剩余次数
     *
     * @param shopId 店铺ID
     * @return 推广修改剩余次数
     * @throws ServiceException 服务异常
     */
    public int getResidueDegree(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.getResidueDegree", params);
    }

    /**
     * 设置推广状态
     *
     * @param shopId 店铺ID
     * @param status 推广状态
     * @throws ServiceException 服务异常
     */
    public void updateWagerStatus(long shopId, WagerOpenStatus status) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("status", status);
        call("eleme.cpc.updateWagerStatus", params);
    }

    /**
     * 设置推广出价
     *
     * @param shopId 店铺ID
     * @param bid 要设置的出价(元)
     * @throws ServiceException 服务异常
     */
    public void setWagerGrade(long shopId, double bid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("bid", bid);
        call("eleme.cpc.setWagerGrade", params);
    }

    /**
     * 设置预算
     *
     * @param shopId 店铺ID
     * @param budget 要设置的预算(元)
     * @throws ServiceException 服务异常
     */
    public void setWagerBudget(long shopId, double budget) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("budget", budget);
        call("eleme.cpc.setWagerBudget", params);
    }

    /**
     * 更新自动投放状态
     *
     * @param shopId 店铺ID
     * @param autoStatus 操作状态
     * @param launchHours 小时集合
     * @throws ServiceException 服务异常
     */
    public void updateAutoStatus(long shopId, WagerAutoStatus autoStatus, List<Integer> launchHours) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("autoStatus", autoStatus);
        params.put("launchHours", launchHours);
        call("eleme.cpc.updateAutoStatus", params);
    }

    /**
     * 设置推广速率
     *
     * @param shopId 店铺ID
     * @param wagerSpeedMode 速率类型
     * @throws ServiceException 服务异常
     */
    public void setWagerSpeed(long shopId, WagerSpeedMode wagerSpeedMode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("wagerSpeedMode", wagerSpeedMode);
        call("eleme.cpc.setWagerSpeed", params);
    }

    /**
     * 获取竞价推广实时排名
     *
     * @param shopId 店铺ID
     * @return 排名信息
     * @throws ServiceException 服务异常
     */
    public ORankEffect getActualRanking(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.cpc.getActualRanking", params);
    }

    /**
     * 查询推广效果数据
     *
     * @param shopId 店铺ID
     * @param beginDate 开始时间
     * @param endDate 结束时间
     * @return 推广效果数据
     * @throws ServiceException 服务异常
     */
    public OWagerUVStatsSummary getUVSummary(long shopId, String beginDate, String endDate) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("beginDate", beginDate);
        params.put("endDate", endDate);
        return call("eleme.cpc.getUVSummary", params);
    }

    /**
     * 查询推广点击分布信息
     *
     * @param shopId 店铺ID
     * @param beginDate 开始时间
     * @param endDate 结束时间
     * @return 推广点击分布数据
     * @throws ServiceException 服务异常
     */
    public ORankAndCostSummary getRankAndCostInfo(long shopId, String beginDate, String endDate) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("beginDate", beginDate);
        params.put("endDate", endDate);
        return call("eleme.cpc.getRankAndCostInfo", params);
    }

    /**
     * 获取推广活跃顾客的点击结构
     *
     * @param shopId 店铺ID
     * @param date 时间
     * @return 推广活跃顾客的点击结构
     * @throws ServiceException 服务异常
     */
    public OUserDistribution getUserDistribution(long shopId, String date) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("date", date);
        return call("eleme.cpc.getUserDistribution", params);
    }
}
