package eleme.openapi.sdk.api.service;

import eleme.openapi.sdk.api.annotation.Service;
import eleme.openapi.sdk.api.base.BaseNopService;
import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.oauth.response.Token;
import eleme.openapi.sdk.config.Config;
import eleme.openapi.sdk.api.entity.product.*;
import eleme.openapi.sdk.api.enumeration.product.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.*;

/**
 * 商品服务
 */
@Service("eleme.product")
public class ProductService extends BaseNopService {
    public ProductService(Config config,Token token) {
        super(config, token, ProductService.class);
    }

    /**
     * 发布品牌菜单v2
     *
     * @param shopId 店铺id
     * @param menu 品牌菜单信息
     * @return 品牌菜单信息
     * @throws ServiceException 服务异常
     */
    public OMenu publishMenuV2(Long shopId, OMenu menu) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menu", menu);
        return call("eleme.product.spuMenuV2.publishMenuV2", params);
    }

    /**
     * 发布品牌菜单分组v2
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @param menuGroup 品牌菜单分组信息
     * @return 菜单分组信息
     * @throws ServiceException 服务异常
     */
    public OMenuGroup publishMenuGroupV2(Long shopId, String menuOutCode, OMenuGroup menuGroup) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        params.put("menuGroup", menuGroup);
        return call("eleme.product.spuMenuV2.publishMenuGroupV2", params);
    }

    /**
     * 获取菜单信息v2
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @return 菜单信息
     * @throws ServiceException 服务异常
     */
    public OMenu getMenuV2(Long shopId, String menuOutCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        return call("eleme.product.spuMenuV2.getMenuV2", params);
    }

    /**
     * 同步全量菜单标品到子门店v2
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @param overrideSyncShopSpu 同步门店菜单
     * @return 同步任务结果
     * @throws ServiceException 服务异常
     */
    public OSpuMenuSyncResult syncMenuV2(Long shopId, String menuOutCode, OverrideSyncShopSpu overrideSyncShopSpu) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        params.put("overrideSyncShopSpu", overrideSyncShopSpu);
        return call("eleme.product.spuMenuV2.syncMenuV2", params);
    }

    /**
     * 同步部分菜单标品到子门店v2
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @param partialSyncShopSpu 同步门店菜单
     * @return 同步任务结果
     * @throws ServiceException 服务异常
     */
    public OSpuMenuSyncResult syncPartialMenuV2(Long shopId, String menuOutCode, PartialSyncShopSpu partialSyncShopSpu) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        params.put("partialSyncShopSpu", partialSyncShopSpu);
        return call("eleme.product.spuMenuV2.syncPartialMenuV2", params);
    }

    /**
     * 获取标品菜单同步任务结果v2
     *
     * @param shopId 店铺id
     * @param mainTaskId 主任务id
     * @return 同步任务结果
     * @throws ServiceException 服务异常
     */
    public OSpuSyncTask getSpuMenuSyncTaskResultV2(Long shopId, Long mainTaskId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("mainTaskId", mainTaskId);
        return call("eleme.product.spuMenuV2.getSpuMenuSyncTaskResultV2", params);
    }

    /**
     * 获取同步菜单任务v2
     *
     * @param shopId 店铺id
     * @param taskIds 任务id列表
     * @return 同步任务详情列表
     * @throws ServiceException 服务异常
     */
    public List<OSpuMenuSyncTaskDetail> batchGetSyncTasksV2(Long shopId, List<Long> taskIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("taskIds", taskIds);
        return call("eleme.product.spuMenuV2.batchGetSyncTasksV2", params);
    }

    /**
     * 查询连锁总店菜单及分组信息
     *
     * @param mid 菜单Id
     * @return 菜单信息以及分组信息
     * @throws ServiceException 服务异常
     */
    public ChainMenuWithGroupDTO getMenuWithGroup(String mid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("mid", mid);
        return call("eleme.product.chain.menu.getMenuWithGroup", params);
    }

    /**
     * 分页查询连锁总店下的菜单列表
     *
     * @param offset 分页起始
     * @param limit 一页个数
     * @return 菜单列表
     * @throws ServiceException 服务异常
     */
    public List<ChainMenuDTO> queryMenuByPage(Long offset, Long limit) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("offset", offset);
        params.put("limit", limit);
        return call("eleme.product.chain.menu.queryMenuByPage", params);
    }

    /**
     * 添加连锁总店菜单
     *
     * @param chainMenuBaseDTO 添加的菜单信息
     * @return 添加的菜单信息
     * @throws ServiceException 服务异常
     */
    public ChainMenuDTO createMenu(ChainMenuBaseDTO chainMenuBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("chainMenuBaseDTO", chainMenuBaseDTO);
        return call("eleme.product.chain.menu.createMenu", params);
    }

    /**
     * 更新连锁总店菜单
     *
     * @param mid 菜单Id
     * @param chainMenuBaseDTO 菜单更新信息
     * @return 更新后的菜单信息
     * @throws ServiceException 服务异常
     */
    public ChainMenuDTO updateMenu(String mid, ChainMenuBaseDTO chainMenuBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("mid", mid);
        params.put("chainMenuBaseDTO", chainMenuBaseDTO);
        return call("eleme.product.chain.menu.updateMenu", params);
    }

    /**
     * 删除连锁总店菜单
     *
     * @param mid 菜单Id
     * @throws ServiceException 服务异常
     */
    public void deleteMenu(String mid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("mid", mid);
        call("eleme.product.chain.menu.deleteMenu", params);
    }

    /**
     * 查询店铺下的所有有效渠道商品
     *
     * @param shopId 店铺id
     * @return 店铺下的商品信息列表
     * @throws ServiceException 服务异常
     */
    public List<ChannelItemDTO> listByShopId(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.channel.item.listByShopId", params);
    }

    /**
     * 根据商品id批量查询商品信息
     *
     * @param itemIds 批量商品id列表
     * @return 商品信息列表
     * @throws ServiceException 服务异常
     */
    public List<ChannelItemDTO> listByItemIds(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.channel.item.listByItemIds", params);
    }

    /**
     * 创建渠道商品
     *
     * @param item 创建商品模型
     * @return 创建结果
     * @throws ServiceException 服务异常
     */
    public ChannelItemDTO create(ChannelItemDTO item) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("item", item);
        return call("eleme.product.channel.item.create", params);
    }

    /**
     * 更新渠道商品
     *
     * @param item 更新商品模型
     * @return 更新结果
     * @throws ServiceException 服务异常
     */
    public ChannelItemDTO modify(ChannelItemDTO item) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("item", item);
        return call("eleme.product.channel.item.modify", params);
    }

    /**
     * 删除渠道商品
     *
     * @param itemId 商品Id
     * @throws ServiceException 服务异常
     */
    public void remove(Long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.channel.item.remove", params);
    }

    /**
     * 批量设置售卖状态
     *
     * @param itemSaleStatusMap 商品id和对应的售卖状态集合
     * @return 对应商品id的操作结果
     * @throws ServiceException 服务异常
     */
    public Map<Long,Boolean> batchSetSaleStatus(Map<Long,Integer> itemSaleStatusMap) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemSaleStatusMap", itemSaleStatusMap);
        return call("eleme.product.channel.item.batchSetSaleStatus", params);
    }

    /**
     * 批量设置库存
     *
     * @param skuStockMap 规格id和对应的库存值
     * @return 对应规格id的操作结果
     * @throws ServiceException 服务异常
     */
    public Map<Long,Boolean> batchSetStock(Map<Long,Integer> skuStockMap) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("skuStockMap", skuStockMap);
        return call("eleme.product.channel.item.batchSetStock", params);
    }

    /**
     * 查询连锁总店商品分组
     *
     * @param gid 连锁总店商品分组Id
     * @return 商品分组
     * @throws ServiceException 服务异常
     */
    public ChainGroupDTO getGroup(String gid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        return call("eleme.product.chain.group.getGroup", params);
    }

    /**
     * 查询连锁总店商品分组及商品详情
     *
     * @param gid 连锁总店商品分组Id
     * @return 商品分组包含商品
     * @throws ServiceException 服务异常
     */
    public ChainGroupWithItemDTO getGroupWithItem(String gid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        return call("eleme.product.chain.group.getGroupWithItem", params);
    }

    /**
     * 添加连锁总店商品分组
     *
     * @param mid 菜单Id
     * @param chainGroupBaseDTO 分组创建信息
     * @return 创建的分组信息
     * @throws ServiceException 服务异常
     */
    public ChainGroupDTO createGroup(String mid, ChainGroupBaseDTO chainGroupBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("mid", mid);
        params.put("chainGroupBaseDTO", chainGroupBaseDTO);
        return call("eleme.product.chain.group.createGroup", params);
    }

    /**
     * 批量添加连锁总店商品分组
     *
     * @param mid 菜单Id
     * @param chainGroupBaseDTOs 分组创建信息列表
     * @return 批量创建结果
     * @throws ServiceException 服务异常
     */
    public BatchChainGroupResultDTO batchCreateGroup(String mid, List<ChainGroupBaseDTO> chainGroupBaseDTOs) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("mid", mid);
        params.put("chainGroupBaseDTOs", chainGroupBaseDTOs);
        return call("eleme.product.chain.group.batchCreateGroup", params);
    }

    /**
     * 更新连锁总店商品分组
     *
     * @param gid 连锁总店商品分组Id
     * @param chainGroupBaseDTO 分组更新信息
     * @return 更新后的分组信息
     * @throws ServiceException 服务异常
     */
    public ChainGroupDTO updateGroup(String gid, ChainGroupBaseDTO chainGroupBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        params.put("chainGroupBaseDTO", chainGroupBaseDTO);
        return call("eleme.product.chain.group.updateGroup", params);
    }

    /**
     * 删除连锁总店商品分组
     *
     * @param gid 连锁总店商品分组Id
     * @throws ServiceException 服务异常
     */
    public void deleteGroup(String gid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        call("eleme.product.chain.group.deleteGroup", params);
    }

    /**
     * 查询连锁总店商品规格关联的单店商品规格信息
     *
     * @param pId 连锁总店商品规格Id
     * @return 单店关联的规格信息列表
     * @throws ServiceException 服务异常
     */
    public List<SingleShopSkuDTO> getRelationByPid(String pId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        return call("eleme.product.chain.pid.getRelationByPid", params);
    }

    /**
     * 设置连锁总店商品规格与单店商品规格关系
     *
     * @param pId 连锁总店商品规格Id
     * @param specId 子店商品规格Id
     * @throws ServiceException 服务异常
     */
    public void setPid(String pId, Long specId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        params.put("specId", specId);
        call("eleme.product.chain.pid.setPid", params);
    }

    /**
     * 批量设置连锁总店商品规格与单店商品规格关系
     *
     * @param pId 连锁总店商品规格Id
     * @param specIds 子店商品规格Id列表
     * @return 批量操作的结果
     * @throws ServiceException 服务异常
     */
    public BatchIdResultDTO batchSetPid(String pId, List<Long> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        params.put("specIds", specIds);
        return call("eleme.product.chain.pid.batchSetPid", params);
    }

    /**
     * 解除连锁总店商品规格与单店商品规格关系
     *
     * @param specId 子店的商品规格Id
     * @throws ServiceException 服务异常
     */
    public void deletePidBySpecId(Long specId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specId", specId);
        call("eleme.product.chain.pid.deletePidBySpecId", params);
    }

    /**
     * 批量解除连锁总店商品规格与单店商品规格关系
     *
     * @param specIds 子店的商品规格Id列表
     * @return 批量操作的结果
     * @throws ServiceException 服务异常
     */
    public BatchIdResultDTO batchDeletePidBySpecId(List<Long> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specIds", specIds);
        return call("eleme.product.chain.pid.batchDeletePidBySpecId", params);
    }

    /**
     * 查询店铺商品分类
     *
     * @param shopId 店铺Id
     * @return 商品分类列表
     * @throws ServiceException 服务异常
     */
    public List<OCategory> getShopCategories(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.category.getShopCategories", params);
    }

    /**
     * 查询店铺商品分类，包含二级分类
     *
     * @param shopId 店铺Id
     * @return 商品分类列表
     * @throws ServiceException 服务异常
     */
    public List<OCategory> getShopCategoriesWithChildren(long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.category.getShopCategoriesWithChildren", params);
    }

    /**
     * 查询商品分类详情
     *
     * @param categoryId 商品分类Id
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory getCategory(Long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        return call("eleme.product.category.getCategory", params);
    }

    /**
     * 查询商品分类详情，包含二级分类
     *
     * @param categoryId 商品分类Id
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory getCategoryWithChildren(long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        return call("eleme.product.category.getCategoryWithChildren", params);
    }

    /**
     * 添加商品分类
     *
     * @param shopId 店铺Id
     * @param name 商品分类名称，长度需在50字以内
     * @param description 商品分类描述，长度需在50字以内
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory createCategory(long shopId, String name, String description) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("name", name);
        params.put("description", description);
        return call("eleme.product.category.createCategory", params);
    }

    /**
     * 添加商品分类v2
     *
     * @param shopId 店铺Id
     * @param name 商品分类名称，长度需在50字以内
     * @param description 商品分类描述，长度需在50字以内
     * @param extCode 商品分类外部编号
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory createCategoryV2(long shopId, String name, String description, String extCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("name", name);
        params.put("description", description);
        params.put("extCode", extCode);
        return call("eleme.product.category.createCategoryV2", params);
    }

    /**
     * 添加商品分类，支持二级分类
     *
     * @param shopId 店铺Id
     * @param name 商品分类名称，长度需在50字以内
     * @param parentId 父分类ID，如果没有可以填0
     * @param description 商品分类描述，长度需在50字以内
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory createCategoryWithChildren(long shopId, String name, long parentId, String description) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("name", name);
        params.put("parentId", parentId);
        params.put("description", description);
        return call("eleme.product.category.createCategoryWithChildren", params);
    }

    /**
     * 更新商品分类
     *
     * @param categoryId 商品分类Id
     * @param name 商品分类名称，长度需在50字以内
     * @param description 商品分类描述，长度需在50字以内
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory updateCategory(long categoryId, String name, String description) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("name", name);
        params.put("description", description);
        return call("eleme.product.category.updateCategory", params);
    }

    /**
     * 更新商品分类v2
     *
     * @param categoryId 商品分类Id
     * @param name 商品分类名称，长度需在50字以内
     * @param description 商品分类描述，长度需在50字以内
     * @param extCode 商品分类外部编号
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory updateCategoryV2(long categoryId, String name, String description, String extCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("name", name);
        params.put("description", description);
        params.put("extCode", extCode);
        return call("eleme.product.category.updateCategoryV2", params);
    }

    /**
     * 更新商品分类，包含二级分类
     *
     * @param categoryId 商品分类Id
     * @param name 商品分类名称，长度需在50字以内
     * @param parentId 父分类ID，如果没有可以填0
     * @param description 商品分类描述，长度需在50字以内
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory updateCategoryWithChildren(long categoryId, String name, long parentId, String description) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("name", name);
        params.put("parentId", parentId);
        params.put("description", description);
        return call("eleme.product.category.updateCategoryWithChildren", params);
    }

    /**
     * 删除商品分类
     *
     * @param categoryId 商品分类Id
     * @return 商品分类
     * @throws ServiceException 服务异常
     */
    public OCategory removeCategory(long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        return call("eleme.product.category.removeCategory", params);
    }

    /**
     * 删除商品分类(新版)
     *
     * @param categoryId 商品分类Id
     * @throws ServiceException 服务异常
     */
    public void invalidCategory(Long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        call("eleme.product.category.invalidCategory", params);
    }

    /**
     * 设置分类排序
     *
     * @param shopId 饿了么店铺Id
     * @param categoryIds 需要排序的分类Id
     * @throws ServiceException 服务异常
     */
    public void setCategoryPositions(Long shopId, List<Long> categoryIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryIds", categoryIds);
        call("eleme.product.category.setCategoryPositions", params);
    }

    /**
     * 设置分类排序(新版)
     *
     * @param shopId 饿了么店铺Id
     * @param categoryIds 需要排序的全部一级分类Id
     * @throws ServiceException 服务异常
     */
    public void setCategorySequence(Long shopId, List<Long> categoryIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryIds", categoryIds);
        call("eleme.product.category.setCategorySequence", params);
    }

    /**
     * 设置二级分类排序
     *
     * @param shopId 饿了么店铺Id
     * @param categoryWithChildrenIds 需要排序的父分类Id，及其下属的二级分类ID
     * @throws ServiceException 服务异常
     */
    public void setCategoryPositionsWithChildren(Long shopId, List<CategoryWithChildrenIds> categoryWithChildrenIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryWithChildrenIds", categoryWithChildrenIds);
        call("eleme.product.category.setCategoryPositionsWithChildren", params);
    }

    /**
     * 查询商品后台类目
     *
     * @param shopId 店铺Id
     * @return 商品后台类目列表
     * @throws ServiceException 服务异常
     */
    public List<OBackCategory> getBackCategory(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.category.getBackCategory", params);
    }

    /**
     * 设置分类类型
     *
     * @param shopId 店铺Id
     * @param categoryId 商品分类Id
     * @param categoryType 分类类型
     * @throws ServiceException 服务异常
     */
    public void setCategoryType(Long shopId, Long categoryId, OCategoryType categoryType) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryId", categoryId);
        params.put("categoryType", categoryType);
        call("eleme.product.category.setCategoryType", params);
    }

    /**
     * 设置分组分时段置顶
     *
     * @param shopId 店铺Id
     * @param categoryId 商品分类Id
     * @param dayPartingStick 置顶时间设置
     * @return 商品分组
     * @throws ServiceException 服务异常
     */
    public OCategory setDayPartingStickTime(Long shopId, Long categoryId, DayPartingStick dayPartingStick) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryId", categoryId);
        params.put("dayPartingStick", dayPartingStick);
        return call("eleme.product.category.setDayPartingStickTime", params);
    }

    /**
     * 删除分组的分时置顶功能
     *
     * @param shopId 店铺Id
     * @param categoryId 商品分类Id
     * @return 是否删除成功
     * @throws ServiceException 服务异常
     */
    public boolean removeDayPartingStickTime(Long shopId, Long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("categoryId", categoryId);
        return call("eleme.product.category.removeDayPartingStickTime", params);
    }

    /**
     * 查询新版商品后台类目
     *
     * @param shopId 店铺Id
     * @return 商品后台类目列表
     * @throws ServiceException 服务异常
     */
    public List<OBackCategory> getBackCategoryV2(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.category.getBackCategoryV2", params);
    }

    /**
     * 查询新版商品后台类目属性
     *
     * @param shopId 店铺Id
     * @param backCategoryId 后台类目id
     * @return 商品后台类目属性列表
     * @throws ServiceException 服务异常
     */
    public List<OBackCategoryProperty> getBackCategoryPropertyV2(Long shopId, Long backCategoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("backCategoryId", backCategoryId);
        return call("eleme.product.category.getBackCategoryPropertyV2", params);
    }

    /**
     * 查询新版商品原材料
     *
     * @param shopId 店铺Id
     * @return 商品原材料列表
     * @throws ServiceException 服务异常
     */
    public List<OMaterial> getMaterials(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.category.getMaterials", params);
    }

    /**
     * 添加套餐
     *
     * @param categoryId 分类Id
     * @param oPackage 套餐属性
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem createPackage(long categoryId, OPackage oPackage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("oPackage", oPackage);
        return call("eleme.product.package.createPackage", params);
    }

    /**
     * 修改套餐基本信息
     *
     * @param itemId 新套餐id即OItem中的商品Id
     * @param categoryId 分类Id即OCategory中的分类Id
     * @param update 套餐基本信息
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem updatePackageContent(long itemId, long categoryId, OPackageUpdate update) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("categoryId", categoryId);
        params.put("update", update);
        return call("eleme.product.package.updatePackageContent", params);
    }

    /**
     * 修改套餐和主料的关联关系
     *
     * @param itemId 新套餐id即OItem中的商品Id
     * @param packages 套餐关系
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem updatePackageRelation(long itemId, List<OPackageGroup> packages) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("packages", packages);
        return call("eleme.product.package.updatePackageRelation", params);
    }

    /**
     * 删除套餐
     *
     * @param itemId 套餐Id
     * @throws ServiceException 服务异常
     */
    public void removePackage(long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.package.removePackage", params);
    }

    /**
     * 获取一个分类下的所有商品
     *
     * @param categoryId 商品分类Id
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> getItemsByCategoryId(long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        return call("eleme.product.item.getItemsByCategoryId", params);
    }

    /**
     * 查询商品详情
     *
     * @param itemId 商品Id
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem getItem(long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        return call("eleme.product.item.getItem", params);
    }

    /**
     * 批量查询商品详情
     *
     * @param itemIds 商品Id的列表
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchGetItems(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchGetItems", params);
    }

    /**
     * 添加商品
     *
     * @param categoryId 商品分类Id
     * @param properties 商品属性
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem createItem(long categoryId, Map<OItemCreateProperty,Object> properties) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("properties", properties);
        return call("eleme.product.item.createItem", params);
    }

    /**
     * 批量添加商品
     *
     * @param categoryId 商品分类Id
     * @param items 商品属性的列表
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchCreateItems(long categoryId, List<Map<OItemCreateProperty,Object>> items) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("items", items);
        return call("eleme.product.item.batchCreateItems", params);
    }

    /**
     * 批量添加商品,且忽略异常,专为星巴克开发
     *
     * @param categoryId 商品分类Id
     * @param items 商品属性的列表
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchCreateItemsIgnoreError(long categoryId, List<Map<OItemCreateProperty,Object>> items) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("items", items);
        return call("eleme.product.item.batchCreateItemsIgnoreError", params);
    }

    /**
     * 更新商品
     *
     * @param itemId 商品Id
     * @param categoryId 商品分类Id
     * @param properties 商品属性
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem updateItem(long itemId, long categoryId, Map<OItemUpdateProperty,Object> properties) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("categoryId", categoryId);
        params.put("properties", properties);
        return call("eleme.product.item.updateItem", params);
    }

    /**
     * 批量更新商品
     *
     * @param requests 批量更新请求
     * @return 批量更新对应的详情结果
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateItems(List<OBatchUpdateItemRequest> requests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("requests", requests);
        return call("eleme.product.item.batchUpdateItems", params);
    }

    /**
     * 批量置满库存
     *
     * @param specIds 商品及商品规格的列表
     * @throws ServiceException 服务异常
     */
    public void batchFillStock(List<OItemIdWithSpecIds> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specIds", specIds);
        call("eleme.product.item.batchFillStock", params);
    }

    /**
     * 批量沽清库存
     *
     * @param specIds 商品及商品规格的列表
     * @throws ServiceException 服务异常
     */
    public void batchClearStock(List<OItemIdWithSpecIds> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specIds", specIds);
        call("eleme.product.item.batchClearStock", params);
    }

    /**
     * 批量修改库存详细信息
     *
     * @param shopId 店铺Id
     * @param updateStockRequests 更新规格请求列表
     * @throws ServiceException 服务异常
     */
    public void batchUpdateStockDetail(Long shopId, List<OUpdateStockRequest> updateStockRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateStockRequests", updateStockRequests);
        call("eleme.product.item.batchUpdateStockDetail", params);
    }

    /**
     * 批量上架商品
     *
     * @param specIds 商品及商品规格的列表
     * @throws ServiceException 服务异常
     */
    public void batchOnShelf(List<OItemIdWithSpecIds> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specIds", specIds);
        call("eleme.product.item.batchOnShelf", params);
    }

    /**
     * 批量上架商品(新版)
     *
     * @param itemIds 商品ID列表
     * @return 操作失败列表
     * @throws ServiceException 服务异常
     */
    public OBatchModifiedResult batchListItems(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchListItems", params);
    }

    /**
     * 批量下架商品
     *
     * @param specIds 商品及商品规格的列表
     * @throws ServiceException 服务异常
     */
    public void batchOffShelf(List<OItemIdWithSpecIds> specIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specIds", specIds);
        call("eleme.product.item.batchOffShelf", params);
    }

    /**
     * 批量下架商品(新版)
     *
     * @param itemIds 商品ID列表
     * @return 操作失败列表
     * @throws ServiceException 服务异常
     */
    public OBatchModifiedResult batchDelistItems(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchDelistItems", params);
    }

    /**
     * 删除商品
     *
     * @param itemId 商品Id
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem removeItem(long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        return call("eleme.product.item.removeItem", params);
    }

    /**
     * 删除商品(新版)
     *
     * @param itemId 商品Id
     * @throws ServiceException 服务异常
     */
    public void invalidItem(long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.item.invalidItem", params);
    }

    /**
     * 批量删除商品
     *
     * @param itemIds 商品Id的列表
     * @return 被删除的商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchRemoveItems(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchRemoveItems", params);
    }

    /**
     * 批量删除商品V2
     *
     * @param itemIds 商品Id的列表
     * @return 被删除的商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchRemoveItemsV2(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchRemoveItemsV2", params);
    }

    /**
     * 批量更新商品库存
     *
     * @param specStocks 商品以及规格库存列表
     * @throws ServiceException 服务异常
     */
    public void batchUpdateSpecStocks(List<OItemIdWithSpecStock> specStocks) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specStocks", specStocks);
        call("eleme.product.item.batchUpdateSpecStocks", params);
    }

    /**
     * 批量更新商品库存(新版)
     *
     * @param stockMap 商品规格ID和库存设值的映射
     * @return 操作失败列表
     * @throws ServiceException 服务异常
     */
    public OBatchModifiedResult batchUpdateStock(Map<Long,Integer> stockMap) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("stockMap", stockMap);
        return call("eleme.product.item.batchUpdateStock", params);
    }

    /**
     * 设置商品排序
     *
     * @param categoryId 商品分类Id
     * @param itemIds 商品Id列表
     * @throws ServiceException 服务异常
     */
    public void setItemPositions(Long categoryId, List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("itemIds", itemIds);
        call("eleme.product.item.setItemPositions", params);
    }

    /**
     * 批量沽清库存并在次日2:00开始置满
     *
     * @param clearStocks 店铺Id及商品Id的列表
     * @throws ServiceException 服务异常
     */
    public void clearAndTimingMaxStock(List<ClearStock> clearStocks) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("clearStocks", clearStocks);
        call("eleme.product.item.clearAndTimingMaxStock", params);
    }

    /**
     * 根据商品扩展码获取商品
     *
     * @param shopId 店铺Id
     * @param extendCode 商品扩展码
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem getItemByShopIdAndExtendCode(Long shopId, String extendCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("extendCode", extendCode);
        return call("eleme.product.item.getItemByShopIdAndExtendCode", params);
    }

    /**
     * 根据商品条形码获取商品
     *
     * @param shopId 店铺Id
     * @param barCode 商品条形码
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public List<OItem> getItemsByShopIdAndBarCode(Long shopId, String barCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("barCode", barCode);
        return call("eleme.product.item.getItemsByShopIdAndBarCode", params);
    }

    /**
     * 批量修改商品价格
     *
     * @param shopId 店铺Id
     * @param specPrices 商品Id及其下SkuId和价格对应Map(限制最多50个)
     * @throws ServiceException 服务异常
     */
    public void batchUpdatePrices(Long shopId, List<OItemIdWithSpecPrice> specPrices) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("specPrices", specPrices);
        call("eleme.product.item.batchUpdatePrices", params);
    }

    /**
     * 查询活动商品
     *
     * @param shopId 店铺Id
     * @return 商品ID集合
     * @throws ServiceException 服务异常
     */
    public List<Long> getItemIdsHasActivityByShopId(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getItemIdsHasActivityByShopId", params);
    }

    /**
     * 查询店铺活动商品(新版)
     *
     * @param shopId 店铺Id
     * @return 商品ID集合
     * @throws ServiceException 服务异常
     */
    public List<Long> getShopSalesItems(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getShopSalesItems", params);
    }

    /**
     * 设置订单餐盒费
     *
     * @param shopId  店铺ID
     * @param status 是否按照订单设置餐盒费
     * @param packingFee 订单餐盒费费用
     * @throws ServiceException 服务异常
     */
    public void setOrderPackingFee(Long shopId, boolean status, Double packingFee) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("status", status);
        params.put("packingFee", packingFee);
        call("eleme.product.item.setOrderPackingFee", params);
    }

    /**
     * 分页获取店铺下的商品
     *
     * @param queryPage 分页查询参数
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public List<OItem> queryItemByPage(QueryPage queryPage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("queryPage", queryPage);
        return call("eleme.product.item.queryItemByPage", params);
    }

    /**
     * 分页获取店铺下的商品V2
     *
     * @param queryPage 分页查询参数
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public List<OItem> queryItemByPageV2(QueryPage queryPage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("queryPage", queryPage);
        return call("eleme.product.item.queryItemByPageV2", params);
    }

    /**
     * 获取原材料树（即将下线）
     *
     * @param shopId 店铺ID
     * @return 原材料树
     * @throws ServiceException 服务异常
     */
    public List<MaterialTree> getMaterialTree(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getMaterialTree", params);
    }

    /**
     * 主料关联配料(接口已过期，请勿使用)
     *
     * @param shopId 店铺ID
     * @param mainItemId 主料ID（商品ID）
     * @param ingredientGroup  商品配料分组
     * @throws ServiceException 服务异常
     */
    public void setIngredient(Long shopId, Long mainItemId, IngredientGroup ingredientGroup) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("mainItemId", mainItemId);
        params.put("ingredientGroup", ingredientGroup);
        call("eleme.product.item.setIngredient", params);
    }

    /**
     * 删除配料(接口已过期，请勿使用)
     *
     * @param shopId 店铺ID
     * @param mainItemId 主料ID（商品ID）
     * @throws ServiceException 服务异常
     */
    public void removeIngredient(Long shopId, Long mainItemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("mainItemId", mainItemId);
        call("eleme.product.item.removeIngredient", params);
    }

    /**
     * 针对主菜itemId设置菜品推荐
     *
     * @param shopId 店铺ID
     * @param itemId 商品ID
     * @param relatedItemIds 关联的商品ID
     * @throws ServiceException 服务异常
     */
    public void setRelatedItemIds(Long shopId, Long itemId, List<Long> relatedItemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("itemId", itemId);
        params.put("relatedItemIds", relatedItemIds);
        call("eleme.product.item.setRelatedItemIds", params);
    }

    /**
     * 对主菜itemId设置是否开启菜品推荐
     *
     * @param shopId 店铺ID
     * @param itemId 商品ID
     * @param display 是否展示
     * @return 最终display的值
     * @throws ServiceException 服务异常
     */
    public boolean displayRelatedItemIds(Long shopId, Long itemId, boolean display) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("itemId", itemId);
        params.put("display", display);
        return call("eleme.product.item.displayRelatedItemIds", params);
    }

    /**
     * 针对主菜itemId查询菜品推荐
     *
     * @param shopId 店铺ID
     * @param itemId 商品ID
     * @return 菜品推荐关系
     * @throws ServiceException 服务异常
     */
    public Map<Long,List<Long>> getRelatedItemIds(Long shopId, Long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("itemId", itemId);
        return call("eleme.product.item.getRelatedItemIds", params);
    }

    /**
     * 添加多规格商品
     *
     * @param categoryId 商品分类Id
     * @param properties 商品属性
     * @return 商品
     * @throws ServiceException 服务异常
     */
    public OItem createMultiSpecItem(long categoryId, Map<OItemCreateProperty,Object> properties) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("properties", properties);
        return call("eleme.product.item.createMultiSpecItem", params);
    }

    /**
     * 批量添加多规格商品
     *
     * @param categoryId 商品分类Id
     * @param items 商品属性的列表
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItem> batchCreateMultiSpecItem(long categoryId, List<Map<OItemCreateProperty,Object>> items) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        params.put("items", items);
        return call("eleme.product.item.batchCreateMultiSpecItem", params);
    }

    /**
     * 更新多规格商品
     *
     * @param itemId 商品Id
     * @param categoryId 商品分类Id
     * @param properties 商品属性
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public OItem updateMultiSpecItem(long itemId, long categoryId, Map<OItemUpdateProperty,Object> properties) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("categoryId", categoryId);
        params.put("properties", properties);
        return call("eleme.product.item.updateMultiSpecItem", params);
    }

    /**
     * 批量更新多规格商品
     *
     * @param requests 商品规格
     * @return 批量更新对应的详情结果
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateMultiSpecItem(List<OBatchUpdateItemRequest> requests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("requests", requests);
        return call("eleme.product.item.batchUpdateMultiSpecItem", params);
    }

    /**
     * 设置配料组数据
     *
     * @param itemId 商品Id
     * @param groupRelations 配料组信息
     * @throws ServiceException 服务异常
     */
    public void setIngredientGroup(long itemId, List<OIngredientGroupRelation> groupRelations) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("groupRelations", groupRelations);
        call("eleme.product.item.setIngredientGroup", params);
    }

    /**
     * 设置配料组数据
     *
     * @param specId 商品Id
     * @param groupRelations 配料组信息
     * @throws ServiceException 服务异常
     */
    public void setSkuIngredientGroup(long specId, List<OIngredientGroupRelation> groupRelations) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specId", specId);
        params.put("groupRelations", groupRelations);
        call("eleme.product.item.setSkuIngredientGroup", params);
    }

    /**
     * 删除配料组数据
     *
     * @param itemId 商品Id
     * @throws ServiceException 服务异常
     */
    public void removeIngredientGroup(long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.item.removeIngredientGroup", params);
    }

    /**
     * 获取商品原材料数据（即将下线）
     *
     * @param shopId 店铺ID
     * @return 原材料树
     * @throws ServiceException 服务异常
     */
    public List<MaterialTree> getItemMaterialTree(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getItemMaterialTree", params);
    }

    /**
     * 创建配料组
     *
     * @param ingredientGroup 配料组数据
     * @return 配料组
     * @throws ServiceException 服务异常
     */
    public OIngredientGroupDTO createIngredientGroup(OIngredientGroupDTO ingredientGroup) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ingredientGroup", ingredientGroup);
        return call("eleme.product.item.createIngredientGroup", params);
    }

    /**
     * 批量创建配料组
     *
     * @param ingredientGroups 配料组数据
     * @return 配料组
     * @throws ServiceException 服务异常
     */
    public List<OIngredientGroupDTO> batchCreateIngredientGroups(List<OIngredientGroupDTO> ingredientGroups) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ingredientGroups", ingredientGroups);
        return call("eleme.product.item.batchCreateIngredientGroups", params);
    }

    /**
     * 查询配料组
     *
     * @param ingredientGroupId 配料组id
     * @return 配料组数据
     * @throws ServiceException 服务异常
     */
    public OIngredientGroupDTO getIngredientGroup(Long ingredientGroupId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ingredientGroupId", ingredientGroupId);
        return call("eleme.product.item.getIngredientGroup", params);
    }

    /**
     * 批量查询配料组
     *
     * @param ingredientGroupIds 配料组id列表
     * @return 配料组数据
     * @throws ServiceException 服务异常
     */
    public List<OIngredientGroupDTO> listIngredientGroups(List<Long> ingredientGroupIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ingredientGroupIds", ingredientGroupIds);
        return call("eleme.product.item.listIngredientGroups", params);
    }

    /**
     * 删除配料组
     *
     * @param ingredientGroupId 配料组id
     * @throws ServiceException 服务异常
     */
    public void deleteIngredientGroup(Long ingredientGroupId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ingredientGroupId", ingredientGroupId);
        call("eleme.product.item.deleteIngredientGroup", params);
    }

    /**
     * 给主料商品绑定配料组
     *
     * @param itemId 主料商品id
     * @param ingredientGroupIds 配料组id列表
     * @throws ServiceException 服务异常
     */
    public void bindIngredientGroups(Long itemId, List<Long> ingredientGroupIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("ingredientGroupIds", ingredientGroupIds);
        call("eleme.product.item.bindIngredientGroups", params);
    }

    /**
     * 给主料商品绑定配料组（规格维度）
     *
     * @param specId 主料商品规格id
     * @param ingredientGroupIds 配料组id列表
     * @throws ServiceException 服务异常
     */
    public void bindSkuIngredientGroups(Long specId, List<Long> ingredientGroupIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specId", specId);
        params.put("ingredientGroupIds", ingredientGroupIds);
        call("eleme.product.item.bindSkuIngredientGroups", params);
    }

    /**
     * 解绑配料组
     *
     * @param itemId 主料商品id
     * @param ingredientGroupIds 配料组id列表
     * @throws ServiceException 服务异常
     */
    public void unbindIngredientGroups(Long itemId, List<Long> ingredientGroupIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("ingredientGroupIds", ingredientGroupIds);
        call("eleme.product.item.unbindIngredientGroups", params);
    }

    /**
     * 解绑配料组（规格维度）
     *
     * @param specId 主料商品规格id
     * @param ingredientGroupIds 配料组id列表
     * @throws ServiceException 服务异常
     */
    public void unbindSkuIngredientGroups(Long specId, List<Long> ingredientGroupIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specId", specId);
        params.put("ingredientGroupIds", ingredientGroupIds);
        call("eleme.product.item.unbindSkuIngredientGroups", params);
    }

    /**
     * 移除主料商品的全部配料组
     *
     * @param itemId 主料商品id
     * @throws ServiceException 服务异常
     */
    public void removeMainItemIngredientGroups(Long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.item.removeMainItemIngredientGroups", params);
    }

    /**
     * 移除主料商品的全部配料组（规格维度）
     *
     * @param specId 主料商品规格id
     * @throws ServiceException 服务异常
     */
    public void removeSkuIngredientGroups(Long specId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("specId", specId);
        call("eleme.product.item.removeSkuIngredientGroups", params);
    }

    /**
     * 更新单店商品所属分组
     *
     * @param shopId 店铺id
     * @param itemId 商品id
     * @param categoryId 分类id
     * @throws ServiceException 服务异常
     */
    public void updateItemGroup(Long shopId, Long itemId, Long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("itemId", itemId);
        params.put("categoryId", categoryId);
        call("eleme.product.item.updateItemGroup", params);
    }

    /**
     * 绑定商品视频
     *
     * @param itemId 商品id
     * @param contentId 内容id
     * @throws ServiceException 服务异常
     */
    public void bindVideo(Long itemId, String contentId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("contentId", contentId);
        call("eleme.product.item.bindVideo", params);
    }

    /**
     * 解绑商品视频
     *
     * @param itemId 商品id
     * @throws ServiceException 服务异常
     */
    public void unbindVideo(Long itemId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        call("eleme.product.item.unbindVideo", params);
    }

    /**
     * 获取一个分类下的所有商品V2接口
     *
     * @param categoryId 商品分类Id
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public List<OItem> getItemsByCategoryIdV2(long categoryId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("categoryId", categoryId);
        return call("eleme.product.item.getItemsByCategoryIdV2", params);
    }

    /**
     * 批量更新商品信息
     *
     * @param shopId 店铺id
     * @param request 店铺商品信息
     * @throws ServiceException 服务异常
     */
    public void batchUpdateShopItems(Long shopId, OBatchUpdateShopItemsRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("request", request);
        call("eleme.product.item.batchUpdateShopItems", params);
    }

    /**
     * 批量更新商品配料信息
     *
     * @param shopId 店铺id
     * @param request 店铺商品信息
     * @throws ServiceException 服务异常
     */
    public void batchUpdateItemIngredient(Long shopId, OBatchUpdateItemIngredientRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("request", request);
        call("eleme.product.item.batchUpdateItemIngredient", params);
    }

    /**
     * 批量更新商品类目、类目属性、主材料信息
     *
     * @param shopId 店铺id
     * @param request 店铺商品信息
     * @throws ServiceException 服务异常
     */
    public void batchUpdateItemCategory(Long shopId, OBatchUpdateItemCategoryRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("request", request);
        call("eleme.product.item.batchUpdateItemCategory", params);
    }

    /**
     * 获取店铺内修改受限制的商品，包含两类商品：有锁的商品和有pid关系的商品。这两种商品信息修改受到限制。
     *
     * @param shopId 店铺id
     * @return 修改受限制的商品
     * @throws ServiceException 服务异常
     */
    public List<OItem> getShopLimitedItems(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getShopLimitedItems", params);
    }

    /**
     * 获取店铺内修改受限制的商品，包含两类商品：有锁的商品和有pid关系的商品。这两种商品信息修改受到限制。
     *
     * @param shopId 店铺id
     * @return 修改受限制的商品
     * @throws ServiceException 服务异常
     */
    public OLimitItems getShopLimitedItemsV2(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.item.getShopLimitedItemsV2", params);
    }

    /**
     * 测试专用，获取测试店铺id数据
     *
     * @return 测试店铺id数据
     * @throws ServiceException 服务异常
     */
    public List<Long> getTestShopIds() throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        return call("eleme.product.item.getTestShopIds", params);
    }

    /**
     * 批量查询店铺商品的连锁店PId和锁信息
     *
     * @param itemIds 商品Id列表，数量不超过100
     * @return 每个商品对应的连锁店PId和锁信息
     * @throws ServiceException 服务异常
     */
    public Map<Long,OItemHasPidAndLocks> batchGetPidAndLocks(List<Long> itemIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        return call("eleme.product.item.batchGetPidAndLocks", params);
    }

    /**
     * 批量获取店铺能否支持某能力
     *
     * @param shopIds 店铺ID列表，数量不超过100
     * @param supportCode 支持的能力Code
     * @return 每个店铺对应的code是否支持，默认不支持
     * @throws ServiceException 服务异常
     */
    public Map<Long,Boolean> batchGetShopCanSupportAbility(List<Long> shopIds, String supportCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        params.put("supportCode", supportCode);
        return call("eleme.product.item.batchGetShopCanSupportAbility", params);
    }

    /**
     * 抖音审核回调
     *
     * @param request 回调请求参数
     * @return 是否成功
     * @throws ServiceException 服务异常
     */
    public Boolean dyAuditStatusCallBack(AuditStatusCallBackRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("request", request);
        return call("eleme.product.dy.dyAuditStatusCallBack", params);
    }

    /**
     * 查询单店铺商品诊断数据
     *
     * @param shopId 店铺ID
     * @return 返回值
     * @throws ServiceException 服务异常
     */
    public ShopCommodityDiagDto getShopCommodityDiagnosisData(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.renovate.getShopCommodityDiagnosisData", params);
    }

    /**
     * 一键and批量优化店铺问题商品详情信息
     *
     * @param request 请求
     * @throws ServiceException 服务异常
     */
    public void agreeShopCommodityDiagnosisProblem(AgreeShopDiagnosisRequest request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("request", request);
        call("eleme.product.renovate.agreeShopCommodityDiagnosisProblem", params);
    }

    /**
     * 查询单店铺商品诊断数据 新版
     *
     * @param shopId 店铺ID
     * @return 返回值
     * @throws ServiceException 服务异常
     */
    public ShopCommodityDiagV2Dto getShopCommodityDiagnosisDataV2(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.product.renovate.getShopCommodityDiagnosisDataV2", params);
    }

    /**
     * 一键and批量商品老ID优化店铺问题商品详情信息新版
     *
     * @param request 请求
     * @throws ServiceException 服务异常
     */
    public void agreeShopCommodityDiagnosisProblemV2(AgreeShopDiagnosisV2Request request) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("request", request);
        call("eleme.product.renovate.agreeShopCommodityDiagnosisProblemV2", params);
    }

    /**
     * 查询连锁总店商品信息
     *
     * @param iid 连锁总店商品Id
     * @return 连锁总店商品信息
     * @throws ServiceException 服务异常
     */
    public ChainItemDTO getChainItem(String iid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("iid", iid);
        return call("eleme.product.chain.item.getChainItem", params);
    }

    /**
     * 批量查询连锁总店商品信息
     *
     * @param iids 连锁总店商品Id列表
     * @return 商品信息列表
     * @throws ServiceException 服务异常
     */
    public List<ChainItemDTO> batchGetChainItem(List<String> iids) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("iids", iids);
        return call("eleme.product.chain.item.batchGetChainItem", params);
    }

    /**
     * 添加连锁总店商品
     *
     * @param gid 连锁总店商品分组Id
     * @param chainItemBaseDTO 商品创建信息
     * @return 创建好的商品
     * @throws ServiceException 服务异常
     */
    public ChainItemDTO createChainItem(String gid, ChainItemBaseDTO chainItemBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        params.put("chainItemBaseDTO", chainItemBaseDTO);
        return call("eleme.product.chain.item.createChainItem", params);
    }

    /**
     * 批量添加连锁总店商品
     *
     * @param gid 连锁总店商品分组Id
     * @param chainItemBaseDTOs 商品创建信息列表
     * @return 批量创建结果集
     * @throws ServiceException 服务异常
     */
    public BatchChainItemResultDTO batchCreateChainItem(String gid, List<ChainItemBaseDTO> chainItemBaseDTOs) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        params.put("chainItemBaseDTOs", chainItemBaseDTOs);
        return call("eleme.product.chain.item.batchCreateChainItem", params);
    }

    /**
     * 替换连锁总店商品
     *
     * @param gid 商品分组Id
     * @param chainItemDTO 商品替换信息
     * @return 替换后的商品信息
     * @throws ServiceException 服务异常
     */
    public ChainItemDTO replaceChainItem(String gid, ChainItemDTO chainItemDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        params.put("chainItemDTO", chainItemDTO);
        return call("eleme.product.chain.item.replaceChainItem", params);
    }

    /**
     * 批量替换连锁总店商品
     *
     * @param gid 商品分组Id
     * @param chainItemDTOs 商品替换信息列表
     * @return 批量替换操作结果
     * @throws ServiceException 服务异常
     */
    public BatchReplaceChainItemResultDTO batchReplaceChainItem(String gid, List<ChainItemDTO> chainItemDTOs) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("gid", gid);
        params.put("chainItemDTOs", chainItemDTOs);
        return call("eleme.product.chain.item.batchReplaceChainItem", params);
    }

    /**
     * 更新连锁总店商品不包含规格信息
     *
     * @param iid 连锁总店商品Id
     * @param chainItemBaseDTO 商品更新信息
     * @return 更新后的商品信息
     * @throws ServiceException 服务异常
     */
    public ChainItemDTO updateChainItemWithoutSku(String iid, ChainItemBaseDTO chainItemBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("iid", iid);
        params.put("chainItemBaseDTO", chainItemBaseDTO);
        return call("eleme.product.chain.item.updateChainItemWithoutSku", params);
    }

    /**
     * 删除连锁总店商品
     *
     * @param iid 连锁总店商品Id
     * @throws ServiceException 服务异常
     */
    public void deleteChainItem(String iid) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("iid", iid);
        call("eleme.product.chain.item.deleteChainItem", params);
    }

    /**
     * 查询连锁总店商品规格
     *
     * @param pId 连锁总店商品规格Id
     * @return 连锁总店商品规格信息
     * @throws ServiceException 服务异常
     */
    public ChainSkuDTO getSku(String pId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        return call("eleme.product.chain.item.getSku", params);
    }

    /**
     * 新增连锁总店商品规格
     *
     * @param iid 连锁总店商品Id
     * @param chainSkuBaseDTO 添加规格信息
     * @return 新增的规格信息
     * @throws ServiceException 服务异常
     */
    public ChainSkuDTO addSku(String iid, ChainSkuBaseDTO chainSkuBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("iid", iid);
        params.put("chainSkuBaseDTO", chainSkuBaseDTO);
        return call("eleme.product.chain.item.addSku", params);
    }

    /**
     * 更新连锁总店商品规格
     *
     * @param pId 连锁总店商品规格Id
     * @param chainSkuBaseDTO 规格更新信息
     * @return 更新的规格信息
     * @throws ServiceException 服务异常
     */
    public ChainSkuDTO updateSku(String pId, ChainSkuBaseDTO chainSkuBaseDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        params.put("chainSkuBaseDTO", chainSkuBaseDTO);
        return call("eleme.product.chain.item.updateSku", params);
    }

    /**
     * 删除连锁总店商品规格
     *
     * @param pId 连锁总店商品规格Id
     * @throws ServiceException 服务异常
     */
    public void deleteSku(String pId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("pId", pId);
        call("eleme.product.chain.item.deleteSku", params);
    }

    /**
     * 根据商品extCode批量查询商品详情
     *
     * @param shopId 店铺Id
     * @param itemExtCodes 商品extCode列表，长度最大100
     * @return 商品列表
     * @throws ServiceException 服务异常
     */
    public Map<String,OItem> getItemsByExtCode(Long shopId, List<String> itemExtCodes) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("itemExtCodes", itemExtCodes);
        return call("eleme.product.itemV2.getItemsByExtCode", params);
    }

    /**
     * 根据规格extCode批量修改库存详细信息
     *
     * @param shopId 店铺Id
     * @param updateStockRequests 更新规格请求列表，最大长度200
     * @return 修改规格库存结果列表
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateStockDetailByExtCode(Long shopId, List<OUpdateStockByExtCodeRequest> updateStockRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateStockRequests", updateStockRequests);
        return call("eleme.product.itemV2.batchUpdateStockDetailByExtCode", params);
    }

    /**
     * 根据规格extCode批量修改库存值
     *
     * @param shopId 店铺Id
     * @param updateStockRequests 更新规格请求列表，最大长度200
     * @return 修改规格库存结果列表
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateStockByExtCode(Long shopId, List<OUpdateOnlyStockByExtCodeRequest> updateStockRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateStockRequests", updateStockRequests);
        return call("eleme.product.itemV2.batchUpdateStockByExtCode", params);
    }

    /**
     * 根据商品extCode批量修改商品售卖状态
     *
     * @param shopId 店铺Id
     * @param updateSaleStatusRequests 更新规格请求列表，最大长度30
     * @return 修改商品售卖状态结果列表
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateSaleStatusByExtCode(Long shopId, List<OUpdateSaleStatusByExtCodeRequest> updateSaleStatusRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateSaleStatusRequests", updateSaleStatusRequests);
        return call("eleme.product.itemV2.batchUpdateSaleStatusByExtCode", params);
    }

    /**
     * 根据商品extCode批量修改商品售卖属性状态
     *
     * @param shopId 店铺Id
     * @param updateAttributeSaleStatusRequests 更新商品售卖属性请求列表，最大长度30
     * @return 修改商品售卖状态结果列表
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateAttributeSaleStatusByExtCode(Long shopId, List<OUpdateAttributeSaleStatusByExtCodeRequest> updateAttributeSaleStatusRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateAttributeSaleStatusRequests", updateAttributeSaleStatusRequests);
        return call("eleme.product.itemV2.batchUpdateAttributeSaleStatusByExtCode", params);
    }

    /**
     * 根据商品extCode批量修改商品售卖属性状态
     *
     * @param shopId 店铺Id
     * @param updateIngredientSaleStatusByExtCodeRequests 更新商品售卖属性请求列表，最大长度30
     * @return 修改商品售卖状态结果列表
     * @throws ServiceException 服务异常
     */
    public List<BatchDetailResultDTO> batchUpdateItemIngredientSaleStatusByExtCode(Long shopId, List<OUpdateIngredientSaleStatusByExtCodeRequest> updateIngredientSaleStatusByExtCodeRequests) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("updateIngredientSaleStatusByExtCodeRequests", updateIngredientSaleStatusByExtCodeRequests);
        return call("eleme.product.itemV2.batchUpdateItemIngredientSaleStatusByExtCode", params);
    }

    /**
     * 发布一个品牌标品
     *
     * @param shopId 店铺id
     * @param spu 品牌标品
     * @param isAutoSync 是否自动同步
     * @return SPU发布结果
     * @throws ServiceException 服务异常
     */
    public SpuSyncResult publishSpu(Long shopId, OSpu spu, Boolean isAutoSync) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spu", spu);
        params.put("isAutoSync", isAutoSync);
        return call("eleme.product.spu.publishSpu", params);
    }

    /**
     * 删除一个品牌标品
     *
     * @param shopId 店铺id
     * @param spuOutCode spuOutCode
     * @param isAutoSync 是否自动同步
     * @return SPU删除结果
     * @throws ServiceException 服务异常
     */
    public SpuSyncResult deleteSpu(Long shopId, String spuOutCode, Boolean isAutoSync) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spuOutCode", spuOutCode);
        params.put("isAutoSync", isAutoSync);
        return call("eleme.product.spu.deleteSpu", params);
    }

    /**
     * 修改SPU上下架状态
     *
     * @param shopId 店铺id
     * @param spuOutCode spuOutCode
     * @param saleStatus 售卖状态,0:上架;1:下架
     * @param isAutoSync 是否自动同步
     * @return SPU状态修改结果
     * @throws ServiceException 服务异常
     */
    public OSpu updateSpuSaleStatus(Long shopId, String spuOutCode, Integer saleStatus, Boolean isAutoSync) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spuOutCode", spuOutCode);
        params.put("saleStatus", saleStatus);
        params.put("isAutoSync", isAutoSync);
        return call("eleme.product.spu.updateSpuSaleStatus", params);
    }

    /**
     * 分页获取店铺下的SPU
     *
     * @param queryPage 分页查询参数
     * @return spu列表
     * @throws ServiceException 服务异常
     */
    public List<OSpu> querySpuByPage(QueryPage queryPage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("queryPage", queryPage);
        return call("eleme.product.spu.querySpuByPage", params);
    }

    /**
     * 根据spuOutCode获取spu信息
     *
     * @param shopId 店铺id
     * @param spuOutCode spuOutCode
     * @return 标品信息
     * @throws ServiceException 服务异常
     */
    public OSpu getSpuBySpuOutCode(Long shopId, String spuOutCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spuOutCode", spuOutCode);
        return call("eleme.product.spu.getSpuBySpuOutCode", params);
    }

    /**
     * 发布一个品牌标品v2
     *
     * @param shopId 店铺id
     * @param spu 品牌标品
     * @return      SPU发布结果
     * @throws ServiceException 服务异常
     */
    public OSpu publishSpuV2(Long shopId, OSpu spu) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spu", spu);
        return call("eleme.product.spuV2.publishSpuV2", params);
    }

    /**
     * 删除一个品牌标品v2
     *
     * @param shopId 店铺id
     * @param spuOutCode spuOutCode
     * @return     SPU删除结果
     * @throws ServiceException 服务异常
     */
    public OSpu deleteSpuV2(Long shopId, String spuOutCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spuOutCode", spuOutCode);
        return call("eleme.product.spuV2.deleteSpuV2", params);
    }

    /**
     * 分页获取店铺下的SPUv2
     *
     * @param queryPage 分页查询参数
     * @return spu列表
     * @throws ServiceException 服务异常
     */
    public List<OSpu> querySpuByPageV2(QueryPage queryPage) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("queryPage", queryPage);
        return call("eleme.product.spuV2.querySpuByPageV2", params);
    }

    /**
     * 根据spuOutCode获取spu信息v2
     *
     * @param shopId 店铺id
     * @param spuOutCode spuOutCode
     * @return 标品信息
     * @throws ServiceException 服务异常
     */
    public OSpu getSpuBySpuOutCodeV2(Long shopId, String spuOutCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("spuOutCode", spuOutCode);
        return call("eleme.product.spuV2.getSpuBySpuOutCodeV2", params);
    }

    /**
     * 上传图片，返回图片的hash值
     *
     * @param image 文件内容base64编码值
     * @return 图片的hash值
     * @throws ServiceException 服务异常
     */
    public String uploadImage(String image) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("image", image);
        return call("eleme.file.uploadImage", params);
    }

    /**
     * 通过远程URL上传图片，返回图片的hash值
     *
     * @param url 远程Url地址
     * @return 图片的hash值
     * @throws ServiceException 服务异常
     */
    public String uploadImageWithRemoteUrl(String url) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("url", url);
        return call("eleme.file.uploadImageWithRemoteUrl", params);
    }

    /**
     * 获取上传文件的访问URL，返回文件的Url地址
     *
     * @param hash 图片hash值
     * @return 文件的Url地址
     * @throws ServiceException 服务异常
     */
    public String getUploadedUrl(String hash) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("hash", hash);
        return call("eleme.file.getUploadedUrl", params);
    }

    /**
     * 获取上传图片的url地址(新版)
     *
     * @param hash 图片hash值
     * @return 文件的Url地址
     * @throws ServiceException 服务异常
     */
    public String getImageUrl(String hash) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("hash", hash);
        return call("eleme.file.getImageUrl", params);
    }

    /**
     * 发布品牌菜单
     *
     * @param shopId 店铺id
     * @param menu 品牌菜单信息
     * @return 品牌菜单信息
     * @throws ServiceException 服务异常
     */
    public OMenu publishMenu(Long shopId, OMenu menu) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menu", menu);
        return call("eleme.product.menu.publishMenu", params);
    }

    /**
     * 获取菜单信息
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @return 菜单信息
     * @throws ServiceException 服务异常
     */
    public OMenu getMenu(Long shopId, String menuOutCode) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        return call("eleme.product.menu.getMenu", params);
    }

    /**
     * 同步菜单标品到子门店
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单outCode
     * @param syncShopSpus 同步店铺id列表
     * @return 数据同步任务id
     * @throws ServiceException 服务异常
     */
    public Long syncMenu(Long shopId, String menuOutCode, List<SyncShopSpu> syncShopSpus) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        params.put("syncShopSpus", syncShopSpus);
        return call("eleme.product.menu.syncMenu", params);
    }

    /**
     * 获取同步主任务信息
     *
     * @param shopId 店铺id
     * @param taskId 任务id
     * @return 同步任务详情
     * @throws ServiceException 服务异常
     */
    public OSpuSyncTask getSyncTask(Long shopId, Long taskId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("taskId", taskId);
        return call("eleme.product.menu.getSyncTask", params);
    }

    /**
     * 批量查询同步子任务信息
     *
     * @param shopId 店铺id
     * @param subTaskIds 任务id
     * @return 子任务详情
     * @throws ServiceException 服务异常
     */
    public List<OSpuSyncSubTaskDetail> getSyncSubTasks(Long shopId, Long subTaskIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("subTaskIds", subTaskIds);
        return call("eleme.product.menu.getSyncSubTasks", params);
    }

    /**
     * 覆盖菜单同步预校验
     *
     * @param shopId 店铺id
     * @param menuOutCode 菜单模板outCode
     * @param syncShopSpus 同步店铺id列表
     * @return 预校验结果
     * @throws ServiceException 服务异常
     */
    public OMenuSyncPreCheckResult syncMenuPreCheck(Long shopId, String menuOutCode, List<SyncShopSpu> syncShopSpus) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("menuOutCode", menuOutCode);
        params.put("syncShopSpus", syncShopSpus);
        return call("eleme.product.menu.syncMenuPreCheck", params);
    }
}
