package eleme.openapi.sdk.api.service;

import eleme.openapi.sdk.api.annotation.Service;
import eleme.openapi.sdk.api.base.BaseNopService;
import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.oauth.response.Token;
import eleme.openapi.sdk.config.Config;
import eleme.openapi.sdk.api.entity.shopCreditScore.*;
import eleme.openapi.sdk.api.enumeration.shopCreditScore.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.*;

/**
 * 商户信用分服务
 */
@Service("eleme.shopCreditScore")
public class ShopCreditScoreService extends BaseNopService {
    public ShopCreditScoreService(Config config,Token token) {
        super(config, token, ShopCreditScoreService.class);
    }

    /**
     * 连锁店根据商户ID集合批量查询商户信用分信息
     *
     * @param shopIds 商户ID集合
     * @return 商户信用分信息集合
     * @throws ServiceException 服务异常
     */
    public List<OShopCreditScoreVO> batchQueryShopCreditScores(List<Long> shopIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        return call("eleme.shopCreditScore.chain.batchQueryShopCreditScores", params);
    }

    /**
     * 连锁店根据商户ID集合批量查询店铺权益规则
     *
     * @param shopIds 商户ID集合
     * @return 商户权益方案规则集合
     * @throws ServiceException 服务异常
     */
    public List<OEquityRulesResponseVO> batchQueryShopEquityRules(List<Long> shopIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        return call("eleme.shopCreditScore.chain.batchQueryShopEquityRules", params);
    }

    /**
     * 连锁店根据商户ID集合批量查询店铺扣罚规则
     *
     * @param shopIds 商户ID集合
     * @return 商户扣罚方案规则集合
     * @throws ServiceException 服务异常
     */
    public List<OPunishRulesResponseVO> batchQueryShopPunishRules(List<Long> shopIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        return call("eleme.shopCreditScore.chain.batchQueryShopPunishRules", params);
    }

    /**
     * 连锁店根据商户ID集合批量查询查询商户信用分变更记录
     *
     * @param shopIds 商户ID集合
     * @return 信用分变更记录集合
     * @throws ServiceException 服务异常
     */
    public List<OScoreRecordQueryRespVO> batchQueryShopCreditScoreRecords(List<Long> shopIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        return call("eleme.shopCreditScore.chain.batchQueryShopCreditScoreRecords", params);
    }

    /**
     * 根据商户ID查询商户信用分信息
     *
     * @param shopId 商户ID
     * @return 商户信用分信息
     * @throws ServiceException 服务异常
     */
    public OShopCreditScoreVO getShopCreditScore(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.shopCreditScore.single.getShopCreditScore", params);
    }

    /**
     * 根据商户id查询商户处置记录
     *
     * @param shopId 商户ID
     * @param eventId 事件id
     * @return 商户处罚明细
     * @throws ServiceException 服务异常
     */
    public OShopPunishRecordVO getShopPunishList(Long shopId, String eventId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("eventId", eventId);
        return call("eleme.shopCreditScore.single.getShopPunishList", params);
    }

    /**
     * 根据商户ID查询店铺权益规则
     *
     * @param shopId 商户ID
     * @return 权益方案规则集合
     * @throws ServiceException 服务异常
     */
    public List<OProgramEquityRulesVO> getShopEquityRules(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.shopCreditScore.single.getShopEquityRules", params);
    }

    /**
     * 根据商户ID查询店铺扣罚规则
     *
     * @param shopId 商户ID
     * @return 扣罚方案规则集合
     * @throws ServiceException 服务异常
     */
    public List<OProgramPunishRulesVO> getShopPunishRules(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.shopCreditScore.single.getShopPunishRules", params);
    }

    /**
     * 查询商户信用分变更记录
     *
     * @param shopId 商户ID
     * @return 信用分变更记录集合
     * @throws ServiceException 服务异常
     */
    public List<OShopScoreRecordVO> getShopCreditScoreRecord(Long shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.shopCreditScore.single.getShopCreditScoreRecord", params);
    }
}
