package eleme.openapi.sdk.api.service;

import eleme.openapi.sdk.api.annotation.Service;
import eleme.openapi.sdk.api.base.BaseNopService;
import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.oauth.response.Token;
import eleme.openapi.sdk.config.Config;
import eleme.openapi.sdk.api.entity.ugc.*;
import eleme.openapi.sdk.api.enumeration.ugc.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.*;

/**
 * 订单评论服务
 */
@Service("eleme.ugc")
public class UgcService extends BaseNopService {
    public UgcService(Config config,Token token) {
        super(config, token, UgcService.class);
    }

    /**
     * 获取指定订单的评论
     *
     * @param orderId 订单id
     * @return 评论信息
     * @throws ServiceException 服务异常
     */
    public OpenapiOrderRate getOrderRateByOrderId(long orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.ugc.getOrderRateByOrderId", params);
    }

    /**
     * 获取指定订单的评论
     *
     * @param orderIds 订单id
     * @return  评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getOrderRatesByOrderIds(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.ugc.getOrderRatesByOrderIds", params);
    }

    /**
     * 获取未回复的评论
     *
     * @param orderIds 订单id
     * @return 未回复的评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getUnreplyOrderRatesByOrderIds(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.ugc.getUnreplyOrderRatesByOrderIds", params);
    }

    /**
     * 获取指定店铺的评论
     *
     * @param shopId  餐厅id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return 评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getOrderRatesByShopId(String shopId, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getOrderRatesByShopId", params);
    }

    /**
     * 获取指定店铺的评论
     *
     * @param shopIds 店铺id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return  评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getOrderRatesByShopIds(List<String> shopIds, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getOrderRatesByShopIds", params);
    }

    /**
     * 获取未回复的评论
     *
     * @param shopIds 店铺id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return   未回复的评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getUnreplyOrderRatesByShopIds(List<String> shopIds, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopIds", shopIds);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getUnreplyOrderRatesByShopIds", params);
    }

    /**
     * 获取店铺的满意度评价信息
     *
     * @param shopId  餐厅id
     * @param score 满意度,取值范围为1~5，1为最不满意，5为非常满意
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return  评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiOrderRate> getOrderRatesByShopAndRating(String shopId, int score, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        params.put("score", score);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getOrderRatesByShopAndRating", params);
    }

    /**
     * 获取单个商品的评论
     *
     * @param itemId  商品id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return  评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiItemRate> getItemRatesByItemId(String itemId, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getItemRatesByItemId", params);
    }

    /**
     * 获取多个商品的评论
     *
     * @param itemIds 商品id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return  评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiItemRate> getItemRatesByItemIds(List<String> itemIds, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getItemRatesByItemIds", params);
    }

    /**
     * 获取多个商品未回复的评论
     *
     * @param itemIds 店铺id
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @param offset 页面偏移量
     * @param pageSize 页面大小
     * @return 未回复的评论信息
     * @throws ServiceException 服务异常
     */
    public List<OpenapiItemRate> getUnreplyItemRatesByItemIds(List<String> itemIds, String startTime, String endTime, int offset, int pageSize) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        params.put("offset", offset);
        params.put("pageSize", pageSize);
        return call("eleme.ugc.getUnreplyItemRatesByItemIds", params);
    }

    /**
     * 回复订单未回复的评论
     *
     * @param orderId 订单id
     * @param reply 回复内容
     * @throws ServiceException 服务异常
     */
    public void replyRateByOrderId(String orderId, String reply) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("reply", reply);
        call("eleme.ugc.replyRateByOrderId", params);
    }

    /**
     * 批量回复订单未回复的评论
     *
     * @param orderIds 订单id
     * @param reply 回复信息
     * @throws ServiceException 服务异常
     */
    public void replyCommentByOrderIds(List<String> orderIds, String reply) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        params.put("reply", reply);
        call("eleme.ugc.replyCommentByOrderIds", params);
    }

    /**
     * 回复商品回复的评论
     *
     * @param itemId 商品id
     * @param reply 回复内容
     * @param startTime   开始时间,只能查询最近90天的数据
     * @param endTime   结束时间
     * @throws ServiceException 服务异常
     */
    public void replyRatesByItemId(String itemId, String reply, String startTime, String endTime) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemId", itemId);
        params.put("reply", reply);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        call("eleme.ugc.replyRatesByItemId", params);
    }

    /**
     * 回复多个商品评论
     *
     * @param itemIds 商品d
     * @param reply 回复信息
     * @param startTime 开始时间,只能查询最近90天的数据
     * @param endTime 结束时间
     * @throws ServiceException 服务异常
     */
    public void replyRatesByItemIds(List<String> itemIds, String reply, String startTime, String endTime) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("itemIds", itemIds);
        params.put("reply", reply);
        params.put("startTime", startTime);
        params.put("endTime", endTime);
        call("eleme.ugc.replyRatesByItemIds", params);
    }

    /**
     * 通过rateId和shopId 回复指定类型的评论(推荐)
     *
     * @param rateId 评论编号
     * @param shopId  餐厅id
     * @param replyType 评论类型
     * @param reply 回复的内容
     * @throws ServiceException 服务异常
     */
    public void replyRateByRateIdAndShopId(String rateId, String shopId, ReplyType replyType, String reply) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateId", rateId);
        params.put("shopId", shopId);
        params.put("replyType", replyType);
        params.put("reply", reply);
        call("eleme.ugc.replyRateByRateIdAndShopId", params);
    }

    /**
     * 通过rateIds和shopId 批量回复指定类型的评论(推荐)
     *
     * @param rateIds  评论编号
     * @param shopId  餐厅id
     * @param replyType 评论类型
     * @param reply 回复的内容
     * @throws ServiceException 服务异常
     */
    public void replyRateByRateIdsAndShopId(List<String> rateIds, String shopId, ReplyType replyType, String reply) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateIds", rateIds);
        params.put("shopId", shopId);
        params.put("replyType", replyType);
        params.put("reply", reply);
        call("eleme.ugc.replyRateByRateIdsAndShopId", params);
    }

    /**
     * 通过rateIds和shopId 批量回复指定类型的评论(推荐)V2
     *
     * @param rateIds  评论编号
     * @param shopId  餐厅id
     * @param replyType 评论类型
     * @param reply 回复的内容
     * @param templateId 使用模板ID
     * @throws ServiceException 服务异常
     */
    public void replyRateByRateIdsAndShopIdV2(List<String> rateIds, String shopId, ReplyType replyType, String reply, Long templateId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateIds", rateIds);
        params.put("shopId", shopId);
        params.put("replyType", replyType);
        params.put("reply", reply);
        params.put("templateId", templateId);
        call("eleme.ugc.replyRateByRateIdsAndShopIdV2", params);
    }

    /**
     * 根据订单ID赠送代金券给该订单的评价用户
     *
     * @param orderId  订单编号
     * @param coupon 需要赠送的代金券信息
     * @throws ServiceException 服务异常
     */
    public void sendCouponByOrderId(String orderId, CouponDTO coupon) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        params.put("coupon", coupon);
        call("eleme.ugc.sendCouponByOrderId", params);
    }

    /**
     * 根据订单ID获取该订单评价用户的可赠券状态
     *
     * @param orderId  订单编号
     * @return  当前订单的可赠券状态
     * @throws ServiceException 服务异常
     */
    public GiftCouponStatus getOrderCouponStatus(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.ugc.getOrderCouponStatus", params);
    }

    /**
     * 根据订单ID集合获取该订单的已赠券信息集合
     *
     * @param orderIds 订单编号集合
     * @return 订单的已赠券信息集合
     * @throws ServiceException 服务异常
     */
    public List<GiftCouponDTO> getCouponsByOrderIds(List<String> orderIds) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderIds", orderIds);
        return call("eleme.ugc.getCouponsByOrderIds", params);
    }

    /**
     * 获取店铺的推荐赠送代金券信息
     *
     * @param shopId 餐厅ID
     * @return 店铺的推荐赠送代金券信息
     * @throws ServiceException 服务异常
     */
    public RecommendHistoriesCouponDTO getRecommendCouponByShopId(String shopId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("shopId", shopId);
        return call("eleme.ugc.getRecommendCouponByShopId", params);
    }

    /**
     * 查询评价信息(推荐)
     *
     * @param rateQuery 评价查询参数
     * @return 评价信息
     * @throws ServiceException 服务异常
     */
    public ORateResult getORateResult(ORateQuery rateQuery) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateQuery", rateQuery);
        return call("eleme.ugc.getORateResult", params);
    }

    /**
     * 查询指定用户的评价信息
     *
     * @param rateQuery 评价查询参数
     * @return 评价信息
     * @throws ServiceException 服务异常
     */
    public ORateResult getOUserRateResult(OUserRateQuery rateQuery) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateQuery", rateQuery);
        return call("eleme.ugc.getOUserRateResult", params);
    }

    /**
     * 统计评价信息数量
     *
     * @param rateQuery 评价查询参数
     * @return 评价信息数量
     * @throws ServiceException 服务异常
     */
    public long countORateResult(ORateQuery rateQuery) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateQuery", rateQuery);
        return call("eleme.ugc.countORateResult", params);
    }

    /**
     * 通过rateIds和shopId 回复饿了么星选评论
     *
     * @param rateIds  评论编号(订单维度)
     * @param shopId  饿了么侧餐厅id
     * @param reply 回复的内容
     * @throws ServiceException 服务异常
     */
    public void replyBaiduRate(List<String> rateIds, String shopId, String reply) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateIds", rateIds);
        params.put("shopId", shopId);
        params.put("reply", reply);
        call("eleme.ugc.replyBaiduRate", params);
    }

    /**
     * 根据rateId和shopId 赠送代金券给该饿了么星选评价对应订单的评价用户
     *
     * @param rateId  评论编号(订单维度)
     * @param shopId  餐厅id
     * @param coupon 需要赠送的代金券信息
     * @throws ServiceException 服务异常
     */
    public void sendBaiduCoupon(String rateId, String shopId, CouponDTO coupon) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateId", rateId);
        params.put("shopId", shopId);
        params.put("coupon", coupon);
        call("eleme.ugc.sendBaiduCoupon", params);
    }

    /**
     * 根据rateId和shopId获取该订单评价用户的可赠券状态(推荐)
     *
     * @param rateId  评论编号(订单维度)
     * @param shopId  餐厅id
     * @param rateDataType 评价数据类型
     * @return  当前订单的可赠券状态
     * @throws ServiceException 服务异常
     */
    public GiftCouponStatus getRateCouponStatus(String rateId, String shopId, ORateDataType rateDataType) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("rateId", rateId);
        params.put("shopId", shopId);
        params.put("rateDataType", rateDataType);
        return call("eleme.ugc.getRateCouponStatus", params);
    }

    /**
     * 根据评价编号赠送代金券给评价用户(推荐)
     *
     * @param ratingCouponDTO 赠券所需的参数
     * @throws ServiceException 服务异常
     */
    public void ratingCoupon(ORatingCouponDTO ratingCouponDTO) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("ratingCouponDTO", ratingCouponDTO);
        call("eleme.ugc.ratingCoupon", params);
    }

    /**
     * 获取赠券扩展信息(推荐)
     *
     * @param extendsQueries 评价赠券信息查询条件
     * @return 对应评价的已赠券信息
     * @throws ServiceException 服务异常
     */
    public List<OCouponExtendsDTO> getCouponExtendsInfo(List<OCouponExtendsQuery> extendsQueries) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("extendsQueries", extendsQueries);
        return call("eleme.ugc.getCouponExtendsInfo", params);
    }

    /**
     * 查询店铺评分及各评分因子数据
     *
     * @param supplierId  连锁店id
     * @param shopIds  店铺id集合
     * @param offset  页面偏移量
     * @param limit  页面大小
     * @return 店铺评分及各评分因子数据
     * @throws ServiceException 服务异常
     */
    public List<ShopFactorInfo> getShopFactorInfos(Long supplierId, List<Long> shopIds, int offset, int limit) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("supplierId", supplierId);
        params.put("shopIds", shopIds);
        params.put("offset", offset);
        params.put("limit", limit);
        return call("eleme.ugc.getShopFactorInfos", params);
    }

    /**
     * 查询订单评价详情
     *
     * @param orderId  页面大小
     * @return 订单评价详情
     * @throws ServiceException 服务异常
     */
    public OrderRateDetailDTO getOrderRateDetail(String orderId) throws ServiceException {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put("orderId", orderId);
        return call("eleme.ugc.getOrderRateDetail", params);
    }
}
