/**
 * Alipay.com Inc.
 * Copyright (c) 2004-2020 All Rights Reserved.
 */
package eleme.openapi.sdk.efs;

import eleme.openapi.sdk.api.exception.ServiceException;
import eleme.openapi.sdk.media.Result;
import eleme.openapi.sdk.media.client.impl.DefaultMediaClient;
import eleme.openapi.sdk.media.upload.UploadRequest;
import eleme.openapi.sdk.media.upload.UploadResponse;
import eleme.openapi.sdk.utils.StringUtils;

import java.io.File;

/**
 *
 * @author cloudsher
 * @version : EfsMediaClient.java, v 0.1 2020年12月15日 5:36 下午 cloudsher Exp $
 */
public class EfsMediaClient {

    /**
     * 视频文件最大不能超过300M
     */
    private static final Long VIDEO_MAX_LEN = 300 * 1024 * 1024L;

    private volatile DefaultMediaClient mediaClient;

    /**
     * get media client
     * @param token 上传token
     * @throws ServiceException 获取Client异常
     */
    private void getMediaClient(String token) throws ServiceException {
        if (StringUtils.isEmpty(token)) {
            throw new ServiceException("GET_MEDIA_CLIENT_ERROR", "上传token不能为空");
        }
        if (mediaClient == null) {
            synchronized (this) {
                if (mediaClient == null) {
                    this.mediaClient = new DefaultMediaClient(token);
                }
            }
        }
    }

    /**
     * 初始化MediaClient
     * @param token 上传token
     * @throws ServiceException 服务异常
     */
    public EfsMediaClient(String token) throws ServiceException {
        init(token);
    }

    /**
     * init client
     * @param token 上传token
     * @throws ServiceException 服务异常
     */
    public void init(String token) throws ServiceException {
        getMediaClient(token);
    }

    /**
     * 上传视频
     * @param videoFile 本地文件
     * @return 视频fileId
     * @throws ServiceException 服务异常
     */
    public String uploadVideo(File videoFile) throws ServiceException {
        //check upload req
        checkUploadReq(videoFile);
        //upload
        UploadRequest request = new UploadRequest();
        request.setFile(videoFile);
        Result<UploadResponse> result = mediaClient.upload(request);
        if (result != null && result.isSuccess()) {
            return result.getData().getFileId();
        } else {
            throw new ServiceException("UPLOAD_VIDEO_ERROR", "上传视频失败," + (result != null ? result.getMessage() : ""));
        }
    }

    public void checkUploadReq(File videoFile) throws ServiceException {
        if (mediaClient == null) {
            throw new ServiceException("GET_MEDIA_CLIENT", "请先初始化MediaClient实例");
        }

        if (videoFile == null || !videoFile.exists()) {
            throw new ServiceException("UPLOAD_VIDEO_ERROR", "上传视频文件不存在");
        }

        if (videoFile.length() > VIDEO_MAX_LEN) {
            throw new ServiceException("UPLOAD_VIDEO_ERROR", "视频文件大小不能超过300M");
        }
    }

}