package eleme.openapi.sdk.media.client.impl;

import eleme.openapi.sdk.media.MediaConfiguration;
import eleme.openapi.sdk.media.MediaFile;
import eleme.openapi.sdk.media.Result;
import eleme.openapi.sdk.media.Site;
import eleme.openapi.sdk.media.client.MediaClient;
import eleme.openapi.sdk.media.upload.BaseUploadRequest;
import eleme.openapi.sdk.media.upload.MultipartCancelRequest;
import eleme.openapi.sdk.media.upload.MultipartCompleteRequest;
import eleme.openapi.sdk.media.upload.MultipartCompleteResponse;
import eleme.openapi.sdk.media.upload.MultipartInitRequest;
import eleme.openapi.sdk.media.upload.MultipartInitResponse;
import eleme.openapi.sdk.media.upload.MultipartUploadRequest;
import eleme.openapi.sdk.media.upload.MultipartUploadResponse;
import eleme.openapi.sdk.media.upload.UploadClient;
import eleme.openapi.sdk.media.upload.UploadPolicy;
import eleme.openapi.sdk.media.upload.UploadRequest;
import eleme.openapi.sdk.media.upload.UploadResponse;
import eleme.openapi.sdk.media.upload.UploadTokenClient;
import eleme.openapi.sdk.media.upload.impl.DefaultUploadClient;
import eleme.openapi.sdk.media.upload.impl.StringUtils;

import java.io.File;
import java.io.InputStream;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author jinli Feb 7, 2015
 */
public class DefaultMediaClient implements MediaClient {

    private String             token;
    private MediaConfiguration configuration;

    private UploadTokenClient tokenClient;
    private UploadClient uploadClient;

    private UploadPolicy defaultUploadPolicy;

    private static Map<String, DefaultUploadClient> clientMap = new ConcurrentHashMap<String, DefaultUploadClient>();

    public DefaultMediaClient(String token) {
        this(Site.HANG_ZHOU, token);
    }

    /**
     * 可指定上传站点的client
     * @param site 上传站点
     * @param token 鉴权token
     */
    public DefaultMediaClient(Site site, String token) {
        this.token = token;
        this.configuration = new MediaConfiguration();
        if (site == null || StringUtils.isEmpty(site.getUploadEndpoint())) {
            throw new IllegalArgumentException("请指定上传域名[Site#uploadEndpoint]");
        }
        this.configuration.setUploadEndpoint(site.getUploadEndpoint());

        uploadClient = clientMap.get(configuration.getUploadEndpoint());
        if (uploadClient == null) {
            synchronized (this) {
                uploadClient = new DefaultUploadClient(configuration.getUploadEndpoint());
            }
        }
    }

    @Override
    public Result<MediaFile> upload(String dir, String name, File file) {
        UploadRequest req = new UploadRequest(getDefaultUploadPolicy());
        req.setDir(dir);
        req.setName(name);
        req.setFile(file);
        Result<UploadResponse> r = upload(req);
        UploadResponse resp = r.getData();

        if (!r.isSuccess()) {
            return Result.create(r.getHttpStatus(), r.getRequestId(), r.getCode(), r.getMessage());
        }

        MediaFile mf = new MediaFile();
        mf.setDir(resp.getDir());
        mf.setName(resp.getName());
        mf.setMd5(resp.geteTag());
        mf.setMimeType(resp.getMimeType());
        mf.setSize(resp.getFileSize());
        mf.setCreateDate(resp.getFileModified());
        mf.setModifyDate(resp.getFileModified());
        mf.setUrl(resp.getUrl());
        return Result.create(r.getHttpStatus(), mf);
    }

    @Override
    public Result<MediaFile> upload(String dir, String name, InputStream input, long size) {
        UploadRequest req = new UploadRequest(getDefaultUploadPolicy());
        req.setDir(dir);
        req.setName(name);
        req.setInputStream(input, size);
        Result<UploadResponse> r = upload(req);
        UploadResponse resp = r.getData();

        if (!r.isSuccess()) {
            return Result.create(r.getHttpStatus(), r.getRequestId(), r.getCode(), r.getMessage());
        }

        MediaFile mf = new MediaFile();
        mf.setDir(resp.getDir());
        mf.setName(resp.getName());
        mf.setMd5(resp.geteTag());
        mf.setMimeType(resp.getMimeType());
        mf.setSize(resp.getFileSize());
        mf.setCreateDate(resp.getFileModified());
        mf.setModifyDate(resp.getFileModified());
        mf.setUrl(resp.getUrl());
        return Result.create(r.getHttpStatus(), mf);
    }

    @Override
    public Result<UploadResponse> upload(UploadRequest req) {
        setUploadToken(req);
        return uploadClient.upload(req);
    }

    @Override
    public Result<MultipartInitResponse> multipartInit(MultipartInitRequest req) {
        setUploadToken(req);
        return uploadClient.multipartInit(req);
    }

    @Override
    public Result<MultipartUploadResponse> multipartUpload(MultipartUploadRequest req) {
        setUploadToken(req);
        return uploadClient.multipartUpload(req);
    }

    @Override
    public Result<MultipartCompleteResponse> multipartComplete(MultipartCompleteRequest req) {
        setUploadToken(req);
        return uploadClient.multipartComplete(req);
    }

    @Override
    public Result<Void> multipartCancel(MultipartCancelRequest req) {
        setUploadToken(req);
        return uploadClient.multipartCancel(req);
    }

    @Override
    public void setTraceOn(boolean traceOn) {
        this.uploadClient.setTraceOn(traceOn);
    }

    private void setUploadToken(BaseUploadRequest req) {
        if (req.getToken() == null) {
            if (token != null) {
                req.setToken(token);
            } else {
                req.setToken(tokenClient.getUploadToken(getUploadPolicy(req)));
            }
        }
    }

    protected UploadPolicy getUploadPolicy(BaseUploadRequest req) {
        UploadPolicy uploadPolicy = req.getUploadPolicy() != null ? req.getUploadPolicy() : getDefaultUploadPolicy();
        uploadPolicy.setNamespace(configuration.getNamespace());
        return uploadPolicy;
    }

    protected UploadPolicy getDefaultUploadPolicy() {
        return this.defaultUploadPolicy;
    }


}
