package eleme.openapi.sdk.media.upload.impl;

import eleme.openapi.sdk.media.MediaConfiguration;
import eleme.openapi.sdk.media.Result;
import eleme.openapi.sdk.media.common.EncodeUtil;
import eleme.openapi.sdk.media.common.http.HttpClientBase;
import eleme.openapi.sdk.media.common.http.LengthInputStreamBody;
import eleme.openapi.sdk.media.trace.Reporter;
import eleme.openapi.sdk.media.upload.MultipartCancelRequest;
import eleme.openapi.sdk.media.upload.MultipartCompleteRequest;
import eleme.openapi.sdk.media.upload.MultipartCompleteResponse;
import eleme.openapi.sdk.media.upload.MultipartInitRequest;
import eleme.openapi.sdk.media.upload.MultipartInitResponse;
import eleme.openapi.sdk.media.upload.MultipartUploadRequest;
import eleme.openapi.sdk.media.upload.MultipartUploadResponse;
import eleme.openapi.sdk.media.upload.UploadClient;
import eleme.openapi.sdk.media.upload.UploadRequest;
import eleme.openapi.sdk.media.upload.UploadResponse;
import eleme.openapi.sdk.utils.JacksonUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;

import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @author jinli Feb 5, 2015
 */
public class DefaultUploadClient extends HttpClientBase implements UploadClient {

    private static final ContentType DEFAULT_TEXT   = ContentType.create("text/plain", MediaConfiguration.CHARSET);
    private static final ContentType DEFAULT_BINARY = ContentType.DEFAULT_BINARY;

    private String endPoint = MediaConfiguration.UPLOAD_ENDPOINT;

    public DefaultUploadClient(String endPoint) {
        this.endPoint = endPoint;
    }

    public DefaultUploadClient() {
    }


    @Override
    public Result<UploadResponse> upload(UploadRequest req) {
        if (req.getInputStream() == null && (req.getFile() == null || !req.getFile().isFile())) {
            throw new IllegalArgumentException("content is invalid.");
        }

        InputStream input = getInputStream(req);
        long length = getInputStreamLength(req);

        try {
            HttpPost post = new HttpPost(getApiUrl(MediaConfiguration.UPLOAD_API_UPLOAD));

            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            text("dir", req.getDir(), builder);
            text("name", req.getName(), builder);
            text("md5", req.getMd5(), builder);
            text("size", length, builder);
            text("localFileName", req.getLocalFileName(), builder);
            metavars("meta", req.getMetas(), builder);
            metavars("var", req.getVars(), builder);
            addParameters(req.getExtendPars(), builder);
            inputstream("content", input, length, builder);

            post.setHeader("Authorization", req.getToken());
            post.setHeader("User-Agent", getUserAgent(req.getToken()));
            post.setEntity(builder.build());

            return executeWithUploadTrace(post, req, UploadResponse.class, Reporter.OP.UPLOAD, length);
        } finally {
            // close inputstream if created by req's file.
            if (req.getFile() != null) {
                try {
                    input.close();
                } catch (Exception e) {
                    // quietly
                }
            }
        }
    }

    @Override
    public Result<MultipartInitResponse> multipartInit(MultipartInitRequest req) {
        HttpPost post = new HttpPost(getApiUrl(MediaConfiguration.UPLOAD_API_BLOCK_INIT));

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        text("dir", req.getDir(), builder);
        text("name", req.getName(), builder);
        text("md5", req.getMd5(), builder);
        text("localFileName", req.getLocalFileName(), builder);

        metavars("meta", req.getMetas(), builder);
        metavars("var", req.getVars(), builder);
        addParameters(req.getExtendPars(), builder);
        if(req.getContent() != null) {
            text("size", req.getContentSize(), builder);
            inputstream("content", req.getContent(), req.getContentSize(), builder);
        }

        post.setHeader("Authorization", req.getToken());
        post.setHeader("User-Agent", getUserAgent(req.getToken()));
        post.setEntity(builder.build());


        return executeWithUploadTrace(post, req, MultipartInitResponse.class, Reporter.OP.BLOCK_INIT, req.getContentSize());
    }

    @Override
    public Result<MultipartUploadResponse> multipartUpload(MultipartUploadRequest req) {
        HttpPost post = new HttpPost(getApiUrl(MediaConfiguration.UPLOAD_API_BLOCK_UPLOAD));

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        text("dir", req.getDir(), builder);
        text("name", req.getName(), builder);
        text("md5", req.getMd5(), builder);
        text("size", req.getContentSize(), builder);
        text("id", req.getId(), builder);
        text("uploadId", req.getUploadId(), builder);
        text("partNumber", req.getPartNumber(), builder);
        metavars("meta", req.getMetas(), builder);
        metavars("var", req.getVars(), builder);
        addParameters(req.getExtendPars(), builder);
        inputstream("content", req.getContent(), req.getContentSize(), builder);

        post.setHeader("Authorization", req.getToken());
        post.setHeader("User-Agent", getUserAgent(req.getToken()));
        post.setEntity(builder.build());

        return executeWithUploadTrace(post, req, MultipartUploadResponse.class, Reporter.OP.BLOCK_UPLOAD, req.getContentSize());
    }

    @Override
    public Result<MultipartCompleteResponse> multipartComplete(MultipartCompleteRequest req) {
        HttpPost post = new HttpPost(getApiUrl(MediaConfiguration.UPLOAD_API_BLOCK_COMPLETE));

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        text("dir", req.getDir(), builder);
        text("name", req.getName(), builder);
        text("md5", req.getMd5(), builder);
        text("id", req.getId(), builder);
        text("uploadId", req.getUploadId(), builder);
        metavars("meta", req.getMetas(), builder);
        metavars("var", req.getVars(), builder);
        addParameters(req.getExtendPars(), builder);
        text("parts", EncodeUtil.encodeWithURLSafeBase64(JacksonUtils.obj2json(req.getParts())), builder);

        post.setHeader("Authorization", req.getToken());
        post.setHeader("User-Agent", getUserAgent(req.getToken()));
        post.setEntity(builder.build());

        return executeWithUploadTrace(post, req, MultipartCompleteResponse.class, Reporter.OP.BLOCK_COMPLETE, 0);
    }

    @Override
    public Result<Void> multipartCancel(MultipartCancelRequest req) {
        HttpPost post = new HttpPost(getApiUrl(MediaConfiguration.UPLOAD_API_BLOCK_CANCEL));

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        text("dir", req.getDir(), builder);
        text("name", req.getName(), builder);
        text("id", req.getId(), builder);
        text("uploadId", req.getUploadId(), builder);

        post.setHeader("Authorization", req.getToken());
        post.setHeader("User-Agent", getUserAgent(req.getToken()));
        post.setEntity(builder.build());

        return executeWithUploadTrace(post, req, null, Reporter.OP.BLOCK_CANCEL, 0);
    }


    @Override
    public void setTraceOn(boolean traceOn) {
        Reporter.sharedInstance().setTraceOn(traceOn);
    }

    private void text(String name, Object value, MultipartEntityBuilder builder) {
        if (value != null) {
            builder.addTextBody(name, String.valueOf(value), DEFAULT_TEXT);
        }
    }

    private void inputstream(String name, InputStream in, long contentLength, MultipartEntityBuilder builder) {
        builder.addPart(name, new LengthInputStreamBody(in, DEFAULT_BINARY, name, contentLength));
    }

    private void metavars(String prefix, Map<String, String> kvs, MultipartEntityBuilder builder) {
        if (kvs != null && !kvs.isEmpty()) {
            for (Entry<String, String> kv : kvs.entrySet()) {
                builder.addTextBody(prefix + "-" + kv.getKey(), kv.getValue(), DEFAULT_TEXT);
            }
        }
    }

    private void addParameters(Map<String, Object> kvs, MultipartEntityBuilder builder) {
        if (kvs != null && !kvs.isEmpty()) {
            for (Entry<String, Object> kv : kvs.entrySet()) {
                builder.addTextBody(kv.getKey(), kv.getValue().toString(), DEFAULT_TEXT);
            }
        }
    }

    private InputStream getInputStream(UploadRequest req) {
        try {
            return req.getFile() != null ? new FileInputStream(req.getFile()) : req.getInputStream();
        } catch (Exception e) {
            return null;
        }
    }

    private long getInputStreamLength(UploadRequest req) {
        return req.getFile() != null ? req.getFile().length() : req.getInputLength();
    }

    private String getUserAgent(String token) {
        if (token.startsWith("UPLOAD_AK_TOP")) {
            return "ALIMEDIASDK_JAVA_ELE/" + MediaConfiguration.SDK_VERSION;
        } else {
            return "ALIMEDIASDK_JAVA_CLOUD/" + MediaConfiguration.SDK_VERSION;
        }
    }

    protected String getApiUrl(String api) {
        return getEndPoint() + api;
    }


    public String getEndPoint() {
        return endPoint;
    }

    public void setEndPoint(String endPoint) {
        this.endPoint = endPoint;
    }
}
