/*
 * Decompiled with CFR 0.152.
 */
package org.thymeleaf.cache;

import java.lang.ref.SoftReference;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.cache.ICache;
import org.thymeleaf.cache.ICacheEntryValidityChecker;
import org.thymeleaf.util.Validate;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public final class StandardCache<K, V>
implements ICache<K, V> {
    private static final long REPORT_INTERVAL = 300000L;
    private static final String REPORT_FORMAT = "[THYMELEAF][*][*][*][CACHE_REPORT] %8s elements | %12s puts | %12s gets | %12s hits | %12s misses - [%s]";
    private volatile long lastExecution = System.currentTimeMillis();
    private final String name;
    private final boolean useSoftReferences;
    private final int maxSize;
    private final CacheDataContainer<K, V> dataContainer;
    private final ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker;
    private final boolean traceExecution;
    private final Logger logger;
    private final AtomicLong getCount;
    private final AtomicLong putCount;
    private final AtomicLong hitCount;
    private final AtomicLong missCount;

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, Logger logger) {
        this(name, useSoftReferences, initialCapacity, -1, null, logger);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker, Logger logger) {
        this(name, useSoftReferences, initialCapacity, -1, entryValidityChecker, logger);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, int maxSize, Logger logger) {
        this(name, useSoftReferences, initialCapacity, maxSize, null, logger);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, int maxSize, ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker, Logger logger) {
        Validate.notEmpty(name, "Name cannot be null or empty");
        Validate.isTrue(initialCapacity > 0, "Initial capacity must be > 0");
        Validate.isTrue(maxSize != 0, "Cache max size must be either -1 (no limit) or > 0");
        this.name = name;
        this.useSoftReferences = useSoftReferences;
        this.maxSize = maxSize;
        this.entryValidityChecker = entryValidityChecker;
        this.logger = logger;
        this.traceExecution = logger != null && logger.isTraceEnabled();
        this.dataContainer = new CacheDataContainer(this.name, initialCapacity, maxSize, this.traceExecution, this.logger);
        this.getCount = new AtomicLong(0L);
        this.putCount = new AtomicLong(0L);
        this.hitCount = new AtomicLong(0L);
        this.missCount = new AtomicLong(0L);
        if (this.logger != null) {
            if (this.maxSize < 0) {
                this.logger.debug("[THYMELEAF][CACHE_INITIALIZE] Initializing cache {}. Soft references {}.", (Object)this.name, (Object)(this.useSoftReferences ? "are used" : "not used"));
            } else {
                this.logger.debug("[THYMELEAF][CACHE_INITIALIZE] Initializing cache {}. Max size: {}. Soft references {}.", new Object[]{this.name, this.maxSize, this.useSoftReferences ? "are used" : "not used"});
            }
        }
    }

    @Override
    public void put(K key, V value) {
        this.incrementReportEntity(this.putCount);
        CacheEntry<V> entry = new CacheEntry<V>(value, this.useSoftReferences);
        int newSize = this.dataContainer.put(key, entry);
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][{}][CACHE_ADD][{}] Adding cache entry in cache \"{}\" for key \"{}\". New size is {}.", new Object[]{TemplateEngine.threadIndex(), this.name, newSize, this.name, key, newSize});
        }
        this.outputReportIfNeeded();
    }

    @Override
    public V get(K key) {
        return this.get(key, this.entryValidityChecker);
    }

    @Override
    public V get(K key, ICacheEntryValidityChecker<? super K, ? super V> validityChecker) {
        this.incrementReportEntity(this.getCount);
        CacheEntry<? super V> resultEntry = this.dataContainer.get(key);
        if (resultEntry == null) {
            this.incrementReportEntity(this.missCount);
            if (this.traceExecution) {
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_MISS] Cache miss in cache \"{}\" for key \"{}\".", new Object[]{TemplateEngine.threadIndex(), this.name, this.name, key});
            }
            this.outputReportIfNeeded();
            return null;
        }
        V resultValue = resultEntry.getValueIfStillValid(this.name, key, validityChecker, this.traceExecution, this.logger);
        if (resultValue == null) {
            int newSize = this.dataContainer.remove(key);
            if (this.traceExecution) {
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_REMOVE][{}] Removing cache entry in cache \"{}\" (Entry \"{}\" is not valid anymore). New size is {}.", new Object[]{TemplateEngine.threadIndex(), this.name, newSize, this.name, key, newSize});
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_MISS] Cache miss in cache \"{}\" for key \"{}\".", new Object[]{TemplateEngine.threadIndex(), this.name, this.name, key});
            }
            this.incrementReportEntity(this.missCount);
            this.outputReportIfNeeded();
            return null;
        }
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][{}][CACHE_HIT] Cache hit in cache \"{}\" for key \"{}\".", new Object[]{TemplateEngine.threadIndex(), this.name, this.name, key});
        }
        this.incrementReportEntity(this.hitCount);
        this.outputReportIfNeeded();
        return resultValue;
    }

    public Set<K> keySet() {
        return this.dataContainer.keySet();
    }

    @Override
    public void clear() {
        this.dataContainer.clear();
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][*][{}][CACHE_REMOVE][0] Removing ALL cache entries in cache \"{}\". New size is 0.", new Object[]{TemplateEngine.threadIndex(), this.name, this.name});
        }
    }

    @Override
    public void clearKey(K key) {
        int newSize = this.dataContainer.remove(key);
        if (this.traceExecution && newSize != -1) {
            this.logger.trace("[THYMELEAF][{}][*][{}][CACHE_REMOVE][{}] Removed cache entry in cache \"{}\" for key \"{}\". New size is {}.", new Object[]{TemplateEngine.threadIndex(), this.name, newSize, this.name, key, newSize});
        }
    }

    public String getName() {
        return this.name;
    }

    public boolean hasMaxSize() {
        return this.maxSize > 0;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    public boolean getUseSoftReferences() {
        return this.useSoftReferences;
    }

    public int size() {
        return this.dataContainer.size();
    }

    private void incrementReportEntity(AtomicLong entity) {
        if (this.traceExecution) {
            entity.incrementAndGet();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void outputReportIfNeeded() {
        long currentTime;
        if (this.traceExecution && (currentTime = System.currentTimeMillis()) - this.lastExecution >= 300000L) {
            StandardCache standardCache = this;
            synchronized (standardCache) {
                if (currentTime - this.lastExecution >= 300000L) {
                    this.logger.trace(String.format(REPORT_FORMAT, this.size(), this.putCount.get(), this.getCount.get(), this.hitCount.get(), this.missCount.get(), this.name));
                    this.lastExecution = currentTime;
                }
            }
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    static final class CacheEntry<V> {
        private final SoftReference<V> cachedValueReference;
        private final long creationTimeInMillis;
        private final V cachedValueAnchor;

        CacheEntry(V cachedValue, boolean useSoftReferences) {
            this.cachedValueReference = new SoftReference<V>(cachedValue);
            this.cachedValueAnchor = !useSoftReferences ? cachedValue : null;
            this.creationTimeInMillis = System.currentTimeMillis();
        }

        public <K> V getValueIfStillValid(String cacheMapName, K key, ICacheEntryValidityChecker<? super K, ? super V> checker, boolean traceExecution, Logger logger) {
            V cachedValue = this.cachedValueReference.get();
            if (cachedValue == null) {
                if (traceExecution) {
                    logger.trace("[THYMELEAF][{}][*][{}][CACHE_DELETED_REFERENCES] Some entries at cache \"{}\" seem to have been sacrificed by the Garbage Collector (soft references).", new Object[]{TemplateEngine.threadIndex(), cacheMapName, cacheMapName});
                }
                return null;
            }
            if (checker == null || checker.checkIsValueStillValid(key, cachedValue, this.creationTimeInMillis)) {
                return cachedValue;
            }
            return null;
        }

        public long getCreationTimeInMillis() {
            return this.creationTimeInMillis;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    static final class CacheDataContainer<K, V> {
        private final String name;
        private final boolean sizeLimit;
        private final int maxSize;
        private final boolean traceExecution;
        private final Logger logger;
        private final ConcurrentHashMap<K, CacheEntry<V>> container;
        private final Object[] fifo;
        private int fifoPointer;

        CacheDataContainer(String name, int initialCapacity, int maxSize, boolean traceExecution, Logger logger) {
            this.name = name;
            this.container = new ConcurrentHashMap(initialCapacity);
            this.maxSize = maxSize;
            boolean bl = this.sizeLimit = maxSize >= 0;
            if (this.sizeLimit) {
                this.fifo = new Object[this.maxSize];
                Arrays.fill(this.fifo, null);
            } else {
                this.fifo = null;
            }
            this.fifoPointer = 0;
            this.traceExecution = traceExecution;
            this.logger = logger;
        }

        public CacheEntry<V> get(Object key) {
            return this.container.get(key);
        }

        public Set<K> keySet() {
            return this.container.keySet();
        }

        public int put(K key, CacheEntry<V> value) {
            if (this.traceExecution) {
                return this.putWithTracing(key, value);
            }
            return this.putWithoutTracing(key, value);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private int putWithoutTracing(K key, CacheEntry<V> value) {
            CacheEntry<V> existing = this.container.putIfAbsent(key, value);
            if (existing != null) {
                return -1;
            }
            if (!this.sizeLimit) return -1;
            Object[] objectArray = this.fifo;
            synchronized (this.fifo) {
                Object removedKey = this.fifo[this.fifoPointer];
                if (removedKey != null) {
                    this.container.remove(removedKey);
                }
                this.fifo[this.fifoPointer] = key;
                this.fifoPointer = (this.fifoPointer + 1) % this.maxSize;
                // ** MonitorExit[var4_4] (shouldn't be in output)
                return -1;
            }
        }

        private synchronized int putWithTracing(K key, CacheEntry<V> value) {
            CacheEntry<V> existing = this.container.putIfAbsent(key, value);
            if (existing == null && this.sizeLimit) {
                CacheEntry<V> removed;
                Object removedKey = this.fifo[this.fifoPointer];
                if (removedKey != null && (removed = this.container.remove(removedKey)) != null) {
                    Integer newSize = this.container.size();
                    this.logger.trace("[THYMELEAF][{}][{}][CACHE_REMOVE][{}] Max size exceeded for cache \"{}\". Removing entry for key \"{}\". New size is {}.", new Object[]{TemplateEngine.threadIndex(), this.name, newSize, this.name, removedKey, newSize});
                }
                this.fifo[this.fifoPointer] = key;
                this.fifoPointer = (this.fifoPointer + 1) % this.maxSize;
            }
            return this.container.size();
        }

        public int remove(K key) {
            if (this.traceExecution) {
                return this.removeWithTracing(key);
            }
            return this.removeWithoutTracing(key);
        }

        private int removeWithoutTracing(K key) {
            CacheEntry<V> removed = this.container.remove(key);
            if (removed != null && this.sizeLimit && key != null) {
                for (int i = 0; i < this.maxSize; ++i) {
                    if (!key.equals(this.fifo[i])) continue;
                    this.fifo[i] = null;
                    break;
                }
            }
            return -1;
        }

        private synchronized int removeWithTracing(K key) {
            CacheEntry<V> removed = this.container.remove(key);
            if (removed == null) {
                return -1;
            }
            if (this.sizeLimit && key != null) {
                for (int i = 0; i < this.maxSize; ++i) {
                    if (!key.equals(this.fifo[i])) continue;
                    this.fifo[i] = null;
                    break;
                }
            }
            return this.container.size();
        }

        public void clear() {
            this.container.clear();
        }

        public int size() {
            return this.container.size();
        }
    }
}

